#pragma once

#include <pgg/query/transactional.h>
#include <internal/envelope/query.h>
#include <internal/thread/add_message_to_thread.h>
#include <pgg/query/ranges.h>
#include <pgg/cast.h>
#include <pgg/range.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/store_message.h>
#include <internal/reflection/mid.h>
#include <internal/query/comment.h>
#include <internal/envelope/factory.h>
#include <macs/envelopes_repository.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace macs{
namespace pg{

using PGEnvelope = macs::pg::query::Envelope;

template <typename ConnProvider, typename TransactionPtr>
class StoreMailishMessage: public boost::asio::coroutine,
                           public boost::enable_shared_from_this<StoreMailishMessage<ConnProvider, TransactionPtr>>{
public:
    explicit StoreMailishMessage (ConnProvider conn, TransactionPtr t, pgg::query::RepositoryPtr qr, const std::string& uid,
            const pgg::RequestInfo& ri, const Envelope& entry, const MimeParts& mime, ThreadMeta tm, const LabelSet& labels,
            const FolderSet& folders, macs::EnvelopesRepository::NotificationMode notificationMode, const MailishMessageInfo& mi,
            OnUpdateEnvelope hook, pgg::Milliseconds timeout)
    : connProvider_(conn), transaction_(t), queryRepository_(qr), uid_(uid), requestInfo_(ri), envFactory_(entry),
      mime_(mime), meta_(tm), labels_(labels), folders_(folders), notificationMode_(notificationMode), mailishInfo_(mi),
      hook_(hook), timeout_(timeout)
    {
    }

    void operator()() {
        const auto cb = [this, self=this->shared_from_this()](auto err){
            if (err) {
                hook_(std::move(err));
            } else {
                (*this)();
            }
        };

        reenter (this) {
            yield transaction_->begin(connProvider_, std::move(cb), timeout_);

            yield lock();

            yield checkDuplicates();
            if (!mid_.empty()) {
                yield getStoredMessage();
                yield transaction_->rollback(std::move(cb));
                return hook_(error_code(), UpdateEnvelopeResult(std::move( storedMessage_ ), EnvelopeKind::duplicate));
            }

            yield addToThread();

            yield storeMessage();

            yield getStoredMessage();

            yield transaction_->commit(std::move(cb));

            hook_(macs::error_code(), UpdateEnvelopeResult(std::move( storedMessage_ ), EnvelopeKind::original));
        }
    }

private:

    void lock() {
        using namespace macs::pg::query;
        const auto q = query<LockUserDelivery>();
        const auto hook = [this, self=this->shared_from_this()](auto err){
            if (err) {
                hook_(std::move(err));
            } else {
                (*this)();
            }
        };
        transaction_->execute(q, hook);
    }

    void checkDuplicates() {
        using namespace macs::pg::query;
        const auto q = query<FindMailishDuplicates>(
            query::FolderId(envFactory_.product().fid()),
            query::ImapId(mailishInfo_.externalImapId)
        );
        const auto h = [this, self=this->shared_from_this()](auto err, auto data) {
            this->handleDuplicates(std::move(err), std::move(data));
        };
        transaction_->fetch(q, h);
    }

    void addToThread() {
        const auto h = [this, self=this->shared_from_this()](auto err, const JoinThreadResult& res) {
            if (err) {
                hook_(std::move(err));
            } else {
                applyJoinThreadResult(envFactory_, res);
                (*this)();
            }
        };

        const auto& envelope = envFactory_.product();
        auto addMessageToThread = boost::make_shared<AddMessageToThread<decltype(transaction_)>>(
            transaction_, queryRepository_, uid_, requestInfo_, meta_, envFactory_.product().rfcId(),
            labels_, folders_, envelope.labels(), envelope.fid(), envelope.receiveDate(), h);
        (*addMessageToThread)();
    }

    void storeMessage() {
        using namespace macs::pg::query;
        const auto q = queryUpdate<StoreMessage>(
                PGEnvelope( envFactory_.release() ), mime_, meta_,
                QuietFlag(notificationMode_ == macs::EnvelopesRepository::NotificationMode::off), mailishInfo_);
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleStore(std::move(err), std::move(data));
        };
        transaction_->fetch(q, h);
    }

    void getStoredMessage() {
        using namespace macs::pg::query;
        const auto q = query<MailboxEntriesByIds>( MailIdList( {mid_} ) );
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleGetMessage(std::move(err), std::move(data));
        };
        transaction_->fetch(q, h);
    }

    template <typename DataRange>
    void handleDuplicates(error_code err, DataRange data) {
        if(err) {
            hook_(std::move(err));
        } else {
            if (!data.empty()) {
                mid_ = std::to_string(pgg::cast<reflection::Mid>(data.front()).mid);
            }
            (*this)();
        }
    }

    template <typename DataRange>
    void handleStore(error_code err, DataRange data) {
        if (err) {
            hook_(std::move(err));
        } else if (data.empty()) {
            hook_(error_code(pgg::error::noDataReceived, "No data received as a result of StoreMessage"));
        } else {
            const auto v = pgg::cast< reflection::StoreMessage >( data.front() );
            mid_ = std::to_string( v.mid );
            (*this)();
        }
    }

    template <typename DataRange>
    void handleGetMessage(error_code err, DataRange data) {
        if (err) {
            hook_(std::move(err));
        } else if (data.empty()) {
            hook_(error_code(pgg::error::noDataReceived, "No data received as a result of GetMessageById"));
        } else {
            auto v = pgg::cast< reflection::Envelope >( data.front() );
            storedMessage_ = makeEnvelope(labels_, std::move(v));
            (*this)();
        }
    }

    ConnProvider connProvider_;
    TransactionPtr transaction_;
    pgg::query::RepositoryPtr queryRepository_;
    std::string uid_;
    pgg::RequestInfo requestInfo_;

    macs::EnvelopeFactory envFactory_;
    MimeParts mime_;
    ThreadMeta meta_;
    LabelSet labels_;
    FolderSet folders_;
    macs::EnvelopesRepository::NotificationMode notificationMode_;
    MailishMessageInfo mailishInfo_;
    OnUpdateEnvelope hook_;
    std::string mid_;
    Envelope storedMessage_;
    pgg::Milliseconds timeout_;

    template <typename T, typename ...Args >
    T queryUpdate( Args&& ... args) const {
        return query<T>(requestInfo_, std::forward<Args>(args)...);
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository_, uid_, std::forward<Args>(args)...);
    }
};

}   //namespace pg
}   //namespace macs

#include <boost/asio/unyield.hpp>
