#pragma once

#include <pgg/query/transactional.h>
#include <internal/folder/query.h>
#include <internal/mailish/query.h>
#include <internal/reflection/folder.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {
namespace mailish {

class UpdateFolder {
public:
    UpdateFolder(pgg::query::RepositoryPtr qr, FoldersRepositoryPtr foldersRepository, OnExecute hook, const std::string& uid,
        const Folder& folder, const MailishFolderInfo& mailishInfo, pgg::Milliseconds timeout, FolderSet folders)
        : queryRepository(qr),
          foldersRepository(foldersRepository),
          folders(std::move(folders)),
          hook(std::move(hook)),
          uid(uid),
          folder(folder),
          mailishInfo(mailishInfo),
          transactionTimeout(timeout)
    {
    }

    template <typename Transactional>
    void operator()(pgg::query::Coroutine & coro, Transactional & t) {
        using namespace macs::pg::query;
        auto onError = boost::bind(&UpdateFolder::checkError, this, _1);
        reenter( coro ) {
            yield t.begin(onError, transactionTimeout);
            {
                error_code err = checkCanUpdateFolder();
                if (err) {
                    return hook(std::move(err));
                }
            }
            if (!folder.isSystem()) {
                yield {
                    mailFolderChanged = true;
                    auto q = query<query::UpdateFolder>(FolderId(folder.fid()), FolderName(folder.name()),
                            ParentFolderId(folder.parentId()));
                    t.execute(q, onError);
                }
            }
            yield {
                auto q = query<MailishUpdateFolder>(FolderId(folder.fid()), mailishInfo);
                t.execute(q, onError);
            }
            yield t.commit(onError);
            if (mailFolderChanged) {
                foldersRepository->resetFoldersCache();
            }
            hook(error_code());
        }
    }

    void checkError(error_code err) {
        if (err) {
            hook(err);
        }
    }
private:
    error_code checkCanUpdateFolder() const {
        const auto i = folders.find(folder.fid());
        if (i == folders.end()) {
            return error_code(::macs::error::noSuchFolder, "can't update folder " + folder.fid());
        }
        return folders.checkCanUpdateFolder(folder, i->second);
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), std::forward<Args>(args)...);
    }

    pgg::query::RepositoryPtr queryRepository;
    FoldersRepositoryPtr foldersRepository;
    FolderSet folders;
    OnExecute hook;
    std::string uid;
    Folder folder;
    MailishFolderInfo mailishInfo;
    pgg::Milliseconds transactionTimeout;
    bool mailFolderChanged = false;
};

} // namespace mailish
} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
