#pragma once

#include <internal/service/service.h>
#include <internal/service/shard.h>
#include <macs_pg/service/factory.h>
#include <pgg/database/pool.h>
#include <pgg/database/endpoint_provider.h>
#include <pgg/database/fallback.h>
#include <macs_pg/macs_pg.h>
#include <internal/service/user_journal.h>
#include <boost/optional.hpp>

namespace macs {
namespace pg {

/**
 * Factory for the Service object.
 */
class ServiceFactoryImpl : public macs::pg::ServiceFactory {
public:
    Self & userJournal( user_journal::Journal v ) override {
        journal_ = std::move(v);
        return *this;
    }
    Self & profiler(pgg::profiling::LogPtr v) override {
        profiler_ = v;
        return *this;
    }

    Self& logger(logging::v1::LogPtr v) override {
        logger_ = logging::adaptLog(std::move(v));
        return *this;
    }

    Self& logger(logging::v2::LogPtr v) override {
        logger_ = std::move(v);
        return *this;
    }

    Self & queryConf(QueryConf v) override {
        queryConf_ = v;
        return *this;
    }

    Self & credentials(pgg::Credentials v) override {
        credentials_ = std::move(v);
        return *this;
    }

    Self & autoQueryHandleStartegy( const QueryHandleStrategy & v) override {
        autoQueryHandleStartegy_ = v;
        return *this;
    }

    Self & requestInfo(pgg::RequestInfo v) override {
        requestInfo_ = std::move(v);
        return *this;
    }

    Self & transactionTimeout(Milliseconds tt) override {
        transactionTimeout_ = tt;
        return *this;
    }

    Product product( const Uid & uid, UserType userType = UserType::existing ) const override {
        return mailbox(uid, userType);
    }

    Product mailbox( const Uid & uid, UserType userType = UserType::existing ) const override {
        if(!queryConf_) {
            throw std::invalid_argument("no queryConf specified!");
        }

        auto uidResolver = uidResolverFactory_->product(credentials_, requestInfo_, userType);
        auto generator = createDatabaseGenerator(uid, uidResolver);
        auto userJournal = getUserJournal(uidResolver, pgg::UidResolveParams(uid));
        auto service = boost::make_shared<Service<pgg::fb::DatabaseGenerator>>(std::move(generator), std::move(userJournal),
                                                   uid, queryConf_, requestInfo_, transactionTimeout_, logger_, uidResolver);
        return Product(std::move(service));
    }

    ShardProduct shard( const ShardId & shardId ) const override {
        if(!queryConf_) {
            throw std::invalid_argument("no queryConf specified!");
        }
        if(!shardResolverFactory_) {
            throw std::runtime_error("shard resolver is not set");
        }

        auto shardResolver = shardResolverFactory_->product(credentials_, requestInfo_);
        auto generator = createDatabaseGenerator(shardId, shardResolver);
        auto userJournal = getUserJournal(shardResolver, pgg::ShardResolveParams(shardId));
        auto shard = std::make_shared<Shard<pgg::fb::DatabaseGenerator>>(std::move(generator), std::move(userJournal),
                                             shardId, queryConf_, transactionTimeout_, shardResolver);
        return ShardProduct(std::move(shard));
    }

    Ptr clone() const override {
        return Ptr( new ServiceFactoryImpl(*this));
    }

    ServiceFactoryImpl(pgg::ConnectionPoolPtr connPool, pgg::UidResolverFactoryPtr uidResolverFactory, ShardResolverFactoryPtr shardResolverFactory)
    : connPool(connPool), uidResolverFactory_(uidResolverFactory), shardResolverFactory_(shardResolverFactory) {
    }
private:
    template <typename Resolver>
    static pgg::DatabasePtr createDatabase(const pgg::database::Traits<Resolver>& traits,
                                      pgg::database::LoggingAttributes logAttrs,
                                      pgg::database::EndpointQuery endpoint) {
        using namespace pgg::database;
        using Impl = DatabaseImpl<ResolverProvider<Resolver>>;
        return boost::make_shared<Impl>(ResolverProvider<Resolver>(traits, endpoint), std::move(logAttrs));
    }

    template <typename Id, typename Resolver>
    pgg::fb::DatabaseGenerator createDatabaseGenerator(const Id& id,
                                                  std::shared_ptr<Resolver> resolver) const {
        using ResolverType = std::remove_const_t<Resolver>;

        auto dbf = [
            traits = pgg::database::Traits<ResolverType>(id, connPool, std::move(resolver)),
            profiler = this->profiler_,
            logger = this->logger_
        ]( pgg::database::EndpointQuery endpoint, pgg::LogError logError ){

            return createDatabase( traits, {profiler, logger, logError},
                 std::move(endpoint));
        };

        return pgg::fb::DatabaseGenerator(std::move(dbf), autoQueryHandleStartegy_, logger_);
    }

    template <typename Resolver>
    UserJournalPtr coCreateUserJournal( Resolver resolver, typename Resolver::element_type::Params resolveParams ) const {
        return boost::make_shared<UserJournal<Resolver>>(
            *journal_, std::move(resolver), std::move(resolveParams), logger_);
    }

    template <typename Resolver>
    UserJournalPtr getUserJournal( Resolver resolver, typename Resolver::element_type::Params resolveParams ) const {
        return journal_.is_initialized()
            ? coCreateUserJournal( std::move(resolver), std::move(resolveParams) )
            : UserJournalPtr();
    }

    ConnectionPoolPtr connPool;
    boost::optional<user_journal::Journal> journal_;
    profiling::LogPtr profiler_;
    pgg::query::RepositoryPtr queryConf_;
    UidResolverFactoryPtr uidResolverFactory_;
    ShardResolverFactoryPtr shardResolverFactory_;
    logging::v2::LogPtr logger_;
    Credentials credentials_;
    QueryHandleStrategy autoQueryHandleStartegy_ = readMasterThenReplica;
    RequestInfo requestInfo_;
    Milliseconds transactionTimeout_ = Milliseconds(0);
};

} // namespace pg
} // namespace macs
