#pragma once

#include <macs/types.h>
#include <pgg/query/transactional.h>

#include <boost/asio/yield.hpp>

namespace macs::pg {


class RunInTransaction {
public:
    using CallbackType = std::function<void(macs::ServicePtr, std::function<void(error_code)>)>;
    using ServiceCreator = std::function<macs::ServicePtr(pgg::ConnectionPtr)>;

    RunInTransaction(pgg::Milliseconds timeout, CallbackType cb, OnExecute hook, ServiceCreator sc)
        : timeout(timeout)
        , cb(std::move(cb))
        , hook(std::move(hook))
        , serviceCreator(std::move(sc))
    {}

    template <typename Transactional>
    void operator() (pgg::query::Coroutine& coroutine, Transactional& transactional) {
        const auto beginHandler = [=] (error_code ec, pgg::ConnectionPtr connPtr=nullptr) {
            if (ec) {
                hook(std::move(ec));
            } else {
                this->service = this->serviceCreator(std::move(connPtr));
            }
        };

        const auto errorHandler = [=] (error_code ec) {
            if (ec) {
                hook(std::move(ec));
            }
        };

        const auto handler = [hook = hook, t = transactional.shared_from_this(), errorHandler] (error_code ec) {
            if (ec) {
                hook(std::move(ec));
            } else {
                t->commit(errorHandler);
            }
        };


        reenter (coroutine) {
            yield transactional.begin(beginHandler, timeout);

            yield cb(this->service, handler);
            hook();
        }
    }

private:
    pgg::Milliseconds timeout;
    CallbackType cb;
    OnExecute hook;
    ServiceCreator serviceCreator;
    macs::ServicePtr service;
};

}

#include <boost/asio/unyield.hpp>
