#pragma once

#include <macs_pg/service/service.h>
#include <pgg/database/database.h>
#include <internal/envelope/repository.h>
#include <internal/imap/repository_imap.h>
#include <internal/folder/repository.h>
#include <internal/label/repository.h>
#include <internal/database/database_info.h>
#include <internal/thread/repository.h>
#include <internal/shared_folders/repository.h>
#include <internal/subscribed_folders/repository.h>
#include <internal/subscription/repository.h>
#include <internal/mailish/repository.h>
#include <internal/changelog/repository.h>
#include <internal/tab/repository.h>
#include <internal/collectors/repository.h>
#include <internal/settings/repository.h>
#include <internal/user/repository.h>
#include <internal/backup/repository.h>
#include <internal/sticker/repository.h>
#include <internal/service/run_in_transaction.h>
#include <internal/service/conn_wrapper.h>

namespace macs {
namespace pg {


template <typename DatabaseGenerator, bool AllowTransaction=true>
class Service : public macs::Service {
public:
    Service(DatabaseGenerator dbg, macs::UserJournalPtr journalPtr, const std::string& uidStr,
        pgg::query::RepositoryPtr queryRepositoryPtr, const pgg::RequestInfo& requestInfo,
        pgg::Milliseconds transactionTimeout, logging::v2::LogPtr loggerPtr,
        pgg::UidResolverPtr resolver)
    : journal_(journalPtr), uid_(uidStr),
      queryRepository_(queryRepositoryPtr),
      resolver_(resolver),
      logger_(loggerPtr),
      dbg_(dbg),
      transactionTimeout_(transactionTimeout),
      requestInfo_(requestInfo),

      folders_(createFoldersRepository(journal(), dbg, uid(), queryRepository(), requestInfo, transactionTimeout, logger())),
      labels_(createLabelsRepository(journal(), dbg, uid(), queryRepository(), requestInfo)),
      tabs_(createTabsRepository(dbg, uid(), queryRepository(), requestInfo, logger())),
      envelopes_(createEnvelopesRepository(
          journal(), dbg, labelsRepository(), foldersRepository(), tabsRepository(), uid(), queryRepository(), requestInfo, transactionTimeout)),
      imapRepo_(createImapRepository(journal(), dbg, uid(), queryRepository(), requestInfo)),
      databaseInfo_( createDatabaseInfo(resolver, uid()) ),
      threads_(createThreadsRepository(dbg, labelsRepository(), uid(), queryRepository(), requestInfo)),
      sharedFolders_(createSharedFoldersRepository(dbg, uid(), queryRepository(), requestInfo, transactionTimeout)),
      subscribedFolders_(createSubscribedFoldersRepository(dbg, uid(), labelsRepository(), queryRepository(), requestInfo)),
      subscriptions_(createSubscriptionRepository(dbg, uid(), queryRepository(), transactionTimeout)),
      mailish_(createMailishRepository(dbg, uid(), queryRepository(), requestInfo, foldersRepository(), labelsRepository(), tabsRepository(), transactionTimeout)),
      changelog_(createUserChangeLogRepository(uid(), dbg, queryRepository(), transactionTimeout)),
      collectors_(createCollectorsRepository(uid(), dbg, queryRepository(), transactionTimeout)),
      settings_(createSettingsRepository(dbg, uid(), queryRepository(), requestInfo)),
      users_(createUsersRepository(dbg, uid(), queryRepository(), logger(), requestInfo)),
      backups_(createBackupsRepository(dbg, uid(), queryRepository(), logger(), requestInfo)),
      stickers_(createStickersRepository(journal(), dbg, uid(), queryRepository(), logger(), requestInfo, transactionTimeout))
    { }

    macs::EnvelopesRepository & envelopes() override {
        return *envelopes_;
    }
    const macs::EnvelopesRepository & envelopes() const override {
        return *envelopes_;
    }
    macs::FoldersRepository & folders() override {
        return *folders_;
    }
    const macs::FoldersRepository & folders() const override {
        return *folders_;
    }
    macs::LabelsRepository & labels() override {
        return *labels_;
    }
    const macs::LabelsRepository & labels() const override {
        return *labels_;
    }

    macs::ImapRepository & imapRepo() override {
        return *imapRepo_;
    }
    const macs::ImapRepository & imapRepo() const override {
        return *imapRepo_;
    }

    virtual const macs::DatabaseInfo & databaseInfo() const override {
        return *databaseInfo_;
    }

    macs::DatabaseInfo & databaseInfo() override {
        return *databaseInfo_;
    }

    macs::ThreadsRepository& threads() override {
        return *threads_;
    }

    const macs::ThreadsRepository& threads() const override {
        return *threads_;
    }

    const macs::SharedFoldersRepository& sharedFolders() const override {
        return *sharedFolders_;
    }

    const macs::SubscribedFoldersRepository& subscribedFolders() const override {
        return *subscribedFolders_;
    }

    const macs::SubscriptionRepository& subscriptions() const override {
        return *subscriptions_;
    }

    const macs::MailishRepository& mailish() const override {
        return *mailish_;
    }

    const macs::UserChangeLogRepository& changelog() const override {
        return *changelog_;
    }

    const macs::TabsRepository& tabs() const override {
        return *tabs_;
    }

    const macs::CollectorsRepository& collectors() const override {
        return *collectors_;
    }

    macs::SettingsRepository& settings() override {
        return *settings_;
    }

    const macs::SettingsRepository& settings() const override {
        return *settings_;
    }

    const macs::UsersRepository& users() const override {
        return *users_;
    }

    const macs::BackupsRepository& backups() const override {
        return *backups_;
    }

    const macs::StickersRepository& stickers() const override {
        return *stickers_;
    }

private:

    macs::ServicePtr clone(ConnectionWrapper newDbGen) const {
        if constexpr (AllowTransaction) {

            return boost::make_shared<Service<ConnectionWrapper, false>>(std::move(newDbGen), journal(),
                                                   uid(), queryRepository(), requestInfo_, transactionTimeout_, logger(), resolver_);
        }
        return nullptr;
    }

    void runTransactionalInternal(TransactionBodyFunc func, OnExecute hook) const override {
        auto serviceCreator = [this](pgg::ConnectionPtr conn) {
            return this->clone(ConnectionWrapper{std::move(conn)});
        };

        RunInTransaction rit{transactionTimeout_, std::move(func), hook, std::move(serviceCreator)};
        pgg::query::fallback::runTransactional(dbg_, queryRepository_, std::move(rit));
    }

    UserJournalPtr journal() const { return journal_; }
    const std::string & uid() const { return uid_; }
    pgg::query::RepositoryPtr queryRepository() const { return queryRepository_; }
    LabelsRepositoryPtr labelsRepository() const { return labels_; }
    FoldersRepositoryPtr foldersRepository() const { return folders_; }
    TabsRepositoryPtr tabsRepository() const { return tabs_; }
    logging::v2::LogPtr logger() const { return logger_; }


    macs::UserJournalPtr journal_;
    std::string uid_;
    pgg::query::RepositoryPtr queryRepository_;
    pgg::UidResolverPtr resolver_;
    logging::v2::LogPtr logger_;
    DatabaseGenerator dbg_;
    pgg::Milliseconds transactionTimeout_;
    pgg::RequestInfo requestInfo_;

    FoldersRepositoryPtr folders_;
    LabelsRepositoryPtr labels_;
    TabsRepositoryPtr tabs_;
    EnvelopesRepositoryPtr envelopes_;
    ImapRepositoryPtr imapRepo_;
    DatabaseInfoPtr databaseInfo_;
    ThreadsRepositoryPtr threads_;
    SharedFoldersRepositoryPtr sharedFolders_;
    SubscribedFoldersRepositoryPtr subscribedFolders_;
    SubscriptionRepositoryPtr subscriptions_;
    MailishRepositoryPtr mailish_;
    UserChangeLogRepositoryPtr changelog_;
    CollectorsRepositoryPtr collectors_;
    SettingsRepositoryPtr settings_;
    UsersRepositoryPtr users_;
    BackupsRepositoryPtr backups_;
    StickersRepositoryPtr stickers_;
};

} // namespace pg
} // namespace macs
