#pragma once

#include <macs_pg/service/shard.h>
#include <pgg/service/shard_resolver.h>
#include <pgg/database/database.h>
#include <pgg/query/repository.h>
#include <internal/changelog/repository.h>
#include <internal/subscription/repository.h>
#include <internal/change_queue/repository.h>
#include <internal/mailish_accounts/repository.h>
#include <internal/collectors/shard_repository.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
class Shard : public macs::Shard {
public:
    using ShardId = pgg::ShardResolver::Id;

    Shard(DatabaseGenerator dbg, macs::UserJournalPtr journal_, const ShardId & shardId_,
            pgg::query::RepositoryPtr queryRepository_, pgg::Milliseconds transactionTimeout,
            pgg::ShardResolverPtr resolver)
    : journal_(journal_), shardId_(shardId_), queryRepository_(queryRepository_),
      databaseInfo_( createDatabaseInfo(std::move(resolver), shardId_) ),
      changeLog_( std::make_shared<ChangeLogRepository<DatabaseGenerator>>(dbg, queryRepository(), transactionTimeout) ),
      subscriptions_( std::make_shared<SubscriptionRepository<DatabaseGenerator>>(dbg, queryRepository(), transactionTimeout) ),
      changeQueue_( std::make_shared<ChangeQueueRepository<DatabaseGenerator>>(dbg, queryRepository(), transactionTimeout) ),
      mailishAccounts_( std::make_shared<MailishAccountsRepository<DatabaseGenerator>>(dbg, queryRepository()) ),
      collectors_( std::make_shared<CollectorsShardRepository<DatabaseGenerator>>(dbg, queryRepository()) )
    { }

    const macs::DatabaseInfo & databaseInfo() const override {
        return *databaseInfo_;
    }

    const macs::ChangeLogRepository& changeLog() const override {
        return *changeLog_;
    }

    const macs::SubscriptionRepository& subscriptions() const override {
        return *subscriptions_;
    }

    const macs::ChangeQueueRepository& changeQueue() const override {
        return *changeQueue_;
    }

    const macs::MailishAccountsRepository& mailishAccounts() const override {
        return *mailishAccounts_;
    }

    const macs::CollectorsShardRepository& collectors() const override {
        return *collectors_;
    }

private:
    const ShardId & shardId() const { return shardId_; }
    pgg::query::RepositoryPtr queryRepository() const { return queryRepository_; }

    macs::UserJournalPtr journal_;
    ShardId shardId_;
    pgg::query::RepositoryPtr queryRepository_;
    DatabaseInfoPtr databaseInfo_;

    ChangeLogRepositoryPtr changeLog_;
    SubscriptionRepositoryPtr subscriptions_;
    ChangeQueueRepositoryPtr changeQueue_;
    MailishAccountsRepositoryPtr mailishAccounts_;
    CollectorsShardRepositoryPtr collectors_;
};

} // namespace pg
} // namespace macs
