#pragma once

#include <yamail/data/serialization/yajl.h>

#include <pgg/query/repository.h>
#include <pgg/cast.h>

#include <macs/hooks.h>
#include <macs/settings_repository.h>

#include <internal/settings/query.h>
#include <internal/settings/transform.h>
#include <internal/query/ids.h>
#include <internal/query/comment.h>
#include <internal/reflection/settings.h>
#include <internal/hooks/wrap.h>

#include <boost/range/algorithm/transform.hpp>

namespace macs::pg {

template <typename DatabaseGenerator>
class SettingsRepository : public macs::SettingsRepository {
public:
    SettingsRepository(
            const DatabaseGenerator& db_,
            const std::string & uid_,
            pgg::query::RepositoryPtr queryRepository_,
            const pgg::RequestInfo& requestInfo_)
        : db(db_)
        , uid(uid_)
        , queryRepository(queryRepository_)
        , requestInfo(requestInfo_) {
    }

private:
    template <typename T, typename ...Args>
    T query(Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository, uid,
            std::forward<Args>(args)...);
    }

    void asyncGetSettings(OnQuerySettings hook) const override {
        db()->fetch(query<query::GetSettings>(), [hook = std::move(hook)](error_code ec, auto data) {
            if (!ec) {
                try {
                    std::vector<reflection::Settings> settings;
                    boost::transform(data, std::back_inserter(settings), [](const auto& row) {
                        return pgg::cast<reflection::Settings>(row);
                    });

                    if(settings.empty()) {
                        hook(std::make_shared<SettingsRaw>());
                    } else {
                        hook(pg::getSettings(settings.at(0)));
                    }
                } catch(const std::exception& e) {
                    hook(error_code(macs::error::input, e.what()));
                }
            } else {
                hook(std::move(ec));
            }
        });
    }

    void asyncInitSettings(SettingsRaw settings, Hook<std::int32_t> hook) const override {
        using yamail::data::serialization::toJson;
        db()->fetch(
            query<query::CreateSettings>(query::Settings(toJson(settings).str())),
            wrapHook<reflection::NumModifRows>(std::move(hook), [](auto data) {
                return data.rows;
            })
        );
    }

    void asyncUpdateSettings(SettingsRaw settings, Hook<std::int32_t> hook) const override {
        using yamail::data::serialization::toJson;
        db()->fetch(
            query<query::UpdateSettings>(query::Settings(toJson(settings).str())),
            wrapHook<reflection::NumModifRows>(std::move(hook), [](auto data) {
                return data.rows;
            })
        );
    };

    void asyncDeleteSettings(Hook<std::int32_t> hook) const override {
        db()->fetch(
            query<query::DeleteSettings>(),
            wrapHook<reflection::NumModifRows>(std::move(hook), [](auto data) {
                return data.rows;
            })
        );
    };

    void asyncEraseSettings(SettingsList names, Hook<std::int32_t> hook) const override {
        db()->fetch(
            query<query::EraseSettings>(query::SettingsNames(std::move(names))),
            wrapHook<reflection::NumModifRows>(std::move(hook), [](auto data) {
                return data.rows;
            })
        );
    };

    const DatabaseGenerator db;
    const std::string uid;
    pgg::query::RepositoryPtr queryRepository;
    const pgg::RequestInfo requestInfo;
};

template <typename DatabaseGenerator>
SettingsRepositoryPtr createSettingsRepository(
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo
) {
    return std::make_shared<SettingsRepository<DatabaseGenerator>>(
        dbg, uid, repo, requestInfo);
}

}
