#pragma once

#include <internal/query/comment.h>
#include <pgg/query/transactional.h>
#include <internal/hooks/wrap.h>
#include <internal/shared_folders/query.h>
#include <internal/folder/query.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class CreateSharedFolderWithArchivation {
public:
    CreateSharedFolderWithArchivation(
            pgg::query::RepositoryPtr queryRepository,
            const Uid& uid, Fid fid,
            const Folder::ArchivationType &type,
            uint32_t keep_days, uint32_t max_size,
            const pgg::RequestInfo& requestInfo,
            pgg::Milliseconds timeout,
            OnUpdate hook) :
        queryRepository(std::move(queryRepository)),
        uid(uid), fid(std::move(fid)),
        type(type), keep_days(keep_days), max_size(max_size),
        requestInfo(requestInfo),
        timeout(timeout),
        hook(std::move(hook)),
        revision(NULL_REVISION) {
    }

    template <typename Transactional>
    void operator()(pgg::query::Coroutine& coro, Transactional& t) {
        const auto handler = [=] (error_code ec) {
            if (ec) {
                hook(std::move(ec));
            }
        };
        error_code ec;

        reenter(coro) {
            yield t.begin(handler, timeout);

            yield requestCreateSharedFolder(t);
            yield requestSetFolderArchivation(t);

            yield t.commit(handler);
            hook(std::move(revision));
        }
    }

private:
    pgg::query::RepositoryPtr queryRepository;
    Uid uid;
    Fid fid;
    Folder::ArchivationType type;
    uint32_t keep_days;
    uint32_t max_size;
    pgg::RequestInfo requestInfo;
    pgg::Milliseconds timeout;
    OnUpdate hook;

    Revision revision;

    auto onUpdate() {
        return wrapHook(OnUpdate([&](error_code ec, Revision rev) mutable {
            if (ec) {
                return this->hook(std::move(ec));
            }
            this->revision = std::move(rev);
        }));
    }

    template <typename Transactional>
    void requestCreateSharedFolder(Transactional& t) {
        using namespace macs::pg::query;
        t.fetch(queryUpdate<CreateSharedFolder>(FolderId(fid)),
                onUpdate()
        );
    }

    template <typename Transactional>
    void requestSetFolderArchivation(Transactional& t) {
        using namespace macs::pg::query;
        t.fetch(queryUpdate<SetFolderArchivationRules>(
                    FolderId(fid),
                    ArchivationType(type),
                    ArchivationTtl(keep_days),
                    MaxFolderSize(max_size)),
                onUpdate()
        );
    }

    template <typename T, typename ... Args>
    T queryUpdate(Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository, uid, requestInfo, std::forward<Args>(args) ...);
    }
};

#include <boost/asio/unyield.hpp>

} //namespace pg
} //namespace macs
