#include <macs_pg/subscribed_folders/factory.h>
#include <internal/subscribed_folders/repository.h>
#include <internal/subscribed_folders/query.h>
#include <internal/reflection/subscribed_folder.h>
#include <internal/envelope/repository.h>
#include <internal/envelope/factory.h>
#include <internal/reflection/sync_message.h>
#include <internal/reflection/fid.h>
#include <internal/reflection/synced_imap_id.h>

namespace macs {
namespace pg {

namespace detail {
inline macs::SubscribedFolder subscribedFolderFromReflection(const reflection::SubscribedFolder& v) {
    return macs::SubscribedFolderFactory()
            .uid(std::to_string(v.uid))
            .fid(std::to_string(v.fid))
            .revision(static_cast<Revision::Value>(v.revision))
            .ownerUid(std::to_string(v.owner_uid))
            .ownerFid(std::to_string(v.owner_fid))
            .syncedRevision(static_cast<Revision::Value>(v.synced_revision))
            .created(v.created)
            .release();
}

inline SyncMessage syncMessageFromReflection(const reflection::SyncMessage& v) {
    SyncMessage result;
    result.mid = std::to_string(v.mid);
    result.revision = static_cast<Revision::Value>(v.revision);
    result.ownerMid = std::to_string(v.owner_mid);
    result.ownerRevision = static_cast<Revision::Value>(v.owner_revision);
    return result;
}
} // namespace detail

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncAddFolder(Fid fid,
                            std::string ownerUid,
                            Fid ownerFid,
                            OnSubscribedFolder hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersAddFolder>(
            query::UserId(uid()),
            query::FolderId(fid),
            query::OwnerUserId(ownerUid),
            query::OwnerFolderId(ownerFid)
    );
    db()->fetch(q, wrapHook<reflection::SubscribedFolder>(std::move(hook), detail::subscribedFolderFromReflection));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetFolder(std::string ownerUid,
                                                 Fid ownerFid,
                                                 OnSubscribedFolder hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetFolder>(
            query::UserId(uid()),
            query::OwnerUserId(ownerUid),
            query::OwnerFolderId(ownerFid)
    );
    db()->fetch(q, wrapHook<reflection::SubscribedFolder>(std::move(hook), detail::subscribedFolderFromReflection));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncRemoveFolders(
        std::string ownerUid,
        std::vector<Fid> ownerFids,
        OnFidVecReceive hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersRemoveFolders>(
            query::UserId(uid()),
            query::OwnerUserId(ownerUid),
            query::FolderIdVector(ownerFids)
    );
    db()->fetch(q, wrapHook<reflection::Fid>(std::move(hook), [](auto v){
        return std::to_string(v.fid);
    }));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncDeleteMessages(std::string ownerUid,
        Fid ownerFid,
        MidVec ownerMids,
        Revision ownerRevision,
        OnUpdateMessages hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersDeleteMessages>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid),
        query::OwnerMids(ownerMids),
        query::OwnerRevision(ownerRevision.value())
    );
    db()->fetch(q, wrapHook(std::move(hook)));
}

namespace detail {

template <typename T>
struct BoolFlag;

template <>
struct BoolFlag<query::SubscribedFoldersAddLabels> : std::true_type {};

template <>
struct BoolFlag<query::SubscribedFoldersRemoveLabels> : std::false_type {};

} // namespace detail

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncAddLabels(const std::string& ownerUid,
                                                 const Fid& ownerFid,
                                                 const MidVec& ownerMids,
                                                 const Revision& ownerRevision,
                                                 const std::vector<Label>& labels,
                                                 OnUpdateMessages hook) const {
    const auto params = createChangeLabelsParams(labels, detail::BoolFlag<query::SubscribedFoldersAddLabels>());

    const auto q = queryRepository().template query<query::SubscribedFoldersAddLabels>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid),
        query::OwnerMids(ownerMids),
        query::OwnerRevision(ownerRevision),
        params
    );
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncRemoveLabels(const std::string& ownerUid,
                                                    const Fid& ownerFid,
                                                    const MidVec& ownerMids,
                                                    const Revision& ownerRevision,
                                                    const std::vector<Label>& labels,
                                                    OnUpdateMessages hook) const {
    const auto params = createChangeLabelsParams(labels, detail::BoolFlag<query::SubscribedFoldersRemoveLabels>());

    const auto q = queryRepository().template query<query::SubscribedFoldersRemoveLabels>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid),
        query::OwnerMids(ownerMids),
        query::OwnerRevision(ownerRevision),
        params
    );
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetSyncedRevision(const std::string& ownerId,
                                                         const Fid& ownerFid,
                                                         OnOptRevisionReceive hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetSyncedRevision>(
        query::UserId(uid()),
        query::OwnerUserId(ownerId),
        query::OwnerFolderId(ownerFid)
    );
    db()->fetch(q, wrapHook<reflection::Revision>(std::move(hook), [](auto value) {
        return PGG_NUMERIC_CAST(Revision, value.revision);
    }));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncSyncMessage(const std::string& ownerUid,
                                                   const Envelope& envelope,
                                                   const MimeParts& mime,
                                                   const std::vector<Hash>& referenceHashes,
                                                   const Hash& inReplyToHash,
                                                   NotificationMode notificationMode,
                                                   OnSyncMessage hook) const {
    const auto convertHash = [](const auto& hash) {
        std::uint64_t result;
        if (!boost::conversion::try_lexical_convert(hash, result)) {
            throw std::invalid_argument(std::string(__PRETTY_FUNCTION__) +
                " invalid hash value=\"" + hash + "\"");
        }
        return result;
    };

    using namespace boost::adaptors;

    std::vector<std::uint64_t> referenceHashValues;
    boost::copy(referenceHashes | filtered([](const auto& value) { return !value.empty(); })
                                | transformed(convertHash),
                std::back_inserter(referenceHashValues));

    auto q = queryRepository().template query<query::SubscribedFoldersSyncMessage>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerRevision(envelope.revision().value()),
        query::Envelope(envelope),
        mime,
        query::ReferenceHashes(referenceHashValues),
        query::InReplyToHash(inReplyToHash.empty() ? boost::none
                                                   : boost::make_optional(convertHash(inReplyToHash))),
        query::QuietFlag(notificationMode == NotificationMode::off)
    );

    int64_t imap_id(0);
    if (boost::conversion::try_lexical_convert(envelope.imapId(), imap_id)) {
        q.set(query::imap_id(imap_id));
    }

    db()->fetch(q, wrapHook<reflection::SyncMessage>(std::move(hook), detail::syncMessageFromReflection));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncJoinThreads(const std::string& ownerUid,
                                                   const Fid& ownerFid,
                                                   const ThreadId& ownerTid,
                                                   const std::vector<ThreadId>& ownerJoinTids,
                                                   const Revision& ownerRevision,
                                                   OnUpdateMessages hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersSyncJoinThreads>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid),
        query::OwnerThreadId(ownerTid),
        query::OwnerJoinThreadIds(ownerJoinTids),
        query::OwnerRevision(ownerRevision)
    );

    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetEnvelopes(const std::string& ownerUid,
                                                    const Fid& ownerFid,
                                                    OnEnvelopeReceive hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetEnvelopes>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid)
    );

    labels().getAllLabels([db = db(), hook = std::move(hook), q = std::move(q)]
                          (error_code ec, LabelSet lbls) mutable {
        if (ec) {
            hook(ec);
        } else {
            db->request(q, wrapHook<reflection::Envelope>(std::move(hook),
                    [lbls = std::move(lbls)](auto v){
                return makeEnvelope(lbls, std::move(v));
            }));
        }
    });
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetFoldersByOwner(const std::string &ownerUid,
                                                         OnSubscribedFolders hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetFoldersByOwner>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid)
    );
    db()->fetch(q, wrapHook<reflection::SubscribedFolder>(std::move(hook), detail::subscribedFolderFromReflection));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetSyncedMids(const std::string &ownerUid,
                                                     const Fid &ownerFid,
                                                     std::size_t rowCount,
                                                     OnMidsReceive hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetSyncedMids>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid),
        query::RowCount(rowCount)
    );
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void SubscribedFoldersRepository<DatabaseGenerator>::asyncGetLastSyncedImapId(const std::string& ownerUid,
        const Fid& ownerFid, Hook<int64_t> hook) const {
    const auto q = queryRepository().template query<query::SubscribedFoldersGetLastSyncedImapId>(
        query::UserId(uid()),
        query::OwnerUserId(ownerUid),
        query::OwnerFolderId(ownerFid)
    );
    db()->fetch(q, wrapHook<reflection::SyncedImapId>(std::move(hook),
        [](const reflection::SyncedImapId& v) {return v.synced_imap_id;}));
}

} //namespace pg
} //namespace macs
