#pragma once

#include <macs_pg/subscription/repository.h>

#include <pgg/database/database.h>
#include <pgg/query/repository.h>

namespace macs {
namespace pg {


template <typename DatabaseGenerator>
class SubscriptionRepository : public macs::SubscriptionRepository {
public:
    typedef macs::SubscriptionRepository Base;

    SubscriptionRepository(const DatabaseGenerator& generator,
                           pgg::query::RepositoryPtr queryRepository,
                           pgg::Milliseconds /*transactionTimeout*/)
        : Base(), uid_(boost::none), queryRepository_(queryRepository), db(std::move(generator))
    {}

    SubscriptionRepository(const DatabaseGenerator& generator,
                           const std::string& uid,
                           pgg::query::RepositoryPtr queryRepository,
                           pgg::Milliseconds /*transactionTimeout*/)
        : Base(), uid_(boost::make_optional(uid)),
          queryRepository_(queryRepository), db(std::move(generator))
    {}

    virtual ~SubscriptionRepository() = default;

private:
    // macs methods
    virtual void asyncFindAJob(std::chrono::seconds aliveTimeout,
                               std::string launchId,
                               std::string hostname,
                               std::string workerVersion,
                               OnFoundJob hook) const override;

    virtual void asyncConfirmTheJob(WorkerId workerId,
                                    std::string launchId,
                                    OnJobConfirmation hook) const override;


    virtual void asyncGetByWorker(WorkerId workerId,
                                  OnSubscriptionsChunkReceive hook) const override;

    virtual void asyncGetById(std::string uid,
                              SubscriptionId subscriptionId,
                              OnSubscriptionReceive hook) const override;

    virtual void asyncTransitState(std::string uid,
                                   SubscriptionId subscriptionId,
                                   Action action,
                                   OnSubscriptionReceive hook) const override;

    virtual void asyncTransitState(SubscriptionId subscriptionId,
                                   Action action,
                                   OnSubscriptionReceive hook) const override {
        asyncTransitState(uid(), subscriptionId, action, std::move(hook));
    }

    virtual void asyncMarkFailed(std::string uid,
                                 SubscriptionId subscriptionId,
                                 std::string failReason,
                                 OnSubscriptionReceive hook) const override;

    virtual void asyncGetFreeForWorker(WorkerId workerId,
                                       std::size_t limit,
                                       OnSubscriptionsChunkReceive hook) const override;

    virtual void asyncRelease(std::string uid,
                              SubscriptionId subscriptionId,
                              WorkerId workerId,
                              OnSubscriptionReceive hook) const override;

    virtual void asyncGetByFids(std::string subscriberUid,
                                std::vector<Fid> fids,
                                OnSubscriptionsChunkReceive hook) const override;

    virtual void asyncRemoveChunk(std::vector<SubscriptionId> subscriptionIds,
                                  OnSubscriptionsChunkReceive hook) const override;

    virtual void asyncAddUnsubscribeTask(UnsubscribeTask task,
                                         OnUnsubscribeTaskReceive hook) const override;

    virtual void asyncGetUnsubscribeTasks(std::size_t limit,
                                          std::chrono::seconds aliveTimeout,
                                          OnUnsubscribeTaskVecReceive hook) const override;

    virtual void asyncRemoveSunscriptionsAndTask(UnsubscribeTask task,
                                                 OnSubscriptionsChunkReceive hook) const override;

    const pgg::query::Repository& queryRepository() const {
        return *queryRepository_;
    }
    std::shared_ptr<const SubscriptionRepository> getSelf() const {
        return std::dynamic_pointer_cast<const SubscriptionRepository>(pgg::share(this));
    }

    const std::string& uid() const {
        if (!uid_) {
            throw std::logic_error("Uid not set. Possible use of Shard service instead of Mailbox service");
        }
        return *uid_;
    }

    boost::optional<std::string> uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
};

template <typename DatabaseGenerator>
SubscriptionRepositoryPtr createSubscriptionRepository(
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    pgg::Milliseconds timeout);

} // namespace pg
} // namespace macs

#include "repository.ipp"
