#include <utility>
#include <sstream>

#include <macs_pg/subscription/factory.h>
#include <internal/subscription/repository.h>
#include <internal/subscription/query.h>
#include <internal/hooks/wrap.h>
#include <internal/reflection/subscription.h>
#include <internal/reflection/found_job.h>
#include <internal/reflection/confirmed_job.h>
#include <macs_pg/subscription/unsubscribe_task_factory.h>
#include <internal/reflection/unsubscribe_task.h>

namespace macs {
namespace pg {

namespace detail {
    inline macs::Subscription subscriptionFromReflection(const reflection::Subscription& v) {
        return macs::SubscriptionFactory()
            .uid(std::to_string(v.uid))
            .subscriptionId(v.subscription_id)
            .fid(std::to_string(v.fid))
            .subscriberUid(std::to_string(v.subscriber_uid))
            .workerId(v.worker_id)
            .state(SubscriptionState::fromString(v.state))
            .release();
    }
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncFindAJob(std::chrono::seconds aliveTimeout,
                                           std::string launchId,
                                           std::string hostname,
                                           std::string workerVersion,
                                           OnFoundJob hook) const {
    const auto q = queryRepository().template query<query::FindAJob>(
        query::AliveTimeoutSeconds(aliveTimeout),
        query::LaunchId(launchId),
        query::Hostname(hostname),
        query::WorkerVersion(workerVersion));
    db()->fetch(q, wrapHook<reflection::FoundJob>(
            std::move(hook),
            [](auto v) {return v.worker_id;}));
}


template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncConfirmTheJob(WorkerId workerId,
                                                std::string launchId,
                                                OnJobConfirmation hook) const {
    const auto q = queryRepository().template query<query::ConfirmTheJob>(
        query::WorkerId(workerId),
        query::LaunchId(launchId));
    db()->fetch(q, wrapHook<reflection::ConfirmedJob>(
            std::move(hook),
            [](auto v) {return v.confirmed;}));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncGetByWorker(WorkerId workerId,
                                              OnSubscriptionsChunkReceive hook) const {
    const auto q = queryRepository().template query<query::SubscriptionsByWorkerId>(workerId);
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook), detail::subscriptionFromReflection));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncGetById(std::string uid,
                                          SubscriptionId subscriptionId,
                                          OnSubscriptionReceive hook) const {
    const auto q = queryRepository().template query<query::SubscriptionById>(
        query::UserId(uid),
        query::SubscriptionId(subscriptionId));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook),
            detail::subscriptionFromReflection,
            [=](std::ostream & s){
                s << "SubscriptionRepository::asyncGetById(" <<
                    uid << ", " << subscriptionId << ")"; }));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncTransitState(std::string uid,
                                               SubscriptionId subscriptionId,
                                               Action action,
                                               OnSubscriptionReceive hook) const {
    const auto q = queryRepository().template query<query::TransitSubscriptionState>(
        query::UserId(uid),
        query::SubscriptionId(subscriptionId),
        query::SubscriptionAction(action));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook),
            detail::subscriptionFromReflection,
            [=](std::ostream & s){
                s << "SubscriptionRepository::asyncTransitState(" <<
                    uid << ", " << subscriptionId << "," << action.toString() << ")"; }));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncMarkFailed(std::string uid,
                                             SubscriptionId subscriptionId,
                                             std::string failReason,
                                             OnSubscriptionReceive hook) const {
    const auto q = queryRepository().template query<query::MarkSubscriptionFailed>(
        query::UserId(uid),
        query::SubscriptionId(subscriptionId),
        query::FailReason(failReason));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook),
            detail::subscriptionFromReflection,
            [=](std::ostream & s){
                s << "SubscriptionRepository::asyncMarkFailed(" <<
                    uid << ", " << subscriptionId << "," << failReason << ")"; }));

}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncGetFreeForWorker(WorkerId workerId,
                                                   std::size_t limit,
                                                   OnSubscriptionsChunkReceive hook) const {
    const auto q = queryRepository().template query<query::GetFreeSubscriptions>(
        query::WorkerId(workerId),
        query::SubscriptionsLimit(limit));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook), detail::subscriptionFromReflection));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncRelease(std::string uid,
                                          SubscriptionId subscriptionId,
                                          WorkerId workerId,
                                          OnSubscriptionReceive hook) const {
    const auto q = queryRepository().template query<query::ReleaseSubscription>(
        query::UserId(uid),
        query::SubscriptionId(subscriptionId),
        query::WorkerId(workerId));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook),
            detail::subscriptionFromReflection,
            [=](std::ostream & s){
                s << "SubscriptionRepository::asyncRelease(" <<
                    uid << ", " << subscriptionId << ")"; }));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncGetByFids(std::string subscriberUid,
                                            std::vector<Fid> fids,
                                            OnSubscriptionsChunkReceive hook) const {
    const auto q = queryRepository().template query<query::SubscriptionsBySubscriberAndFids>(
        query::UserId(uid()),
        query::Subscriber(subscriberUid),
        query::FolderIdVector(fids));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook), detail::subscriptionFromReflection));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncRemoveChunk(std::vector<SubscriptionId> subscriptionIds,
                                              OnSubscriptionsChunkReceive hook) const {
    const auto q = queryRepository().template query<query::RemoveSubscriptionsChunk>(
        query::UserId(uid()),
        query::SubscriptionIdVec(subscriptionIds));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook), detail::subscriptionFromReflection));
}

namespace {
    macs::UnsubscribeTask taskFromReflection(const reflection::UnsubscribeTask& v) {
        auto ownerFids = v.owner_fids | boost::adaptors::transformed(
                    &boost::lexical_cast<std::string, std::uint32_t>);
        return macs::UnsubscribeTaskFactory()
            .taskId(static_cast<uint64_t>(v.task_id))
            .taskRequestId(v.task_request_id)
            .ownerUid(std::to_string(v.owner_uid))
            .ownerFids({std::begin(ownerFids), std::end(ownerFids)})
            .subscriberUid(std::to_string(v.subscriber_uid))
            .rootSubscriberFid(std::to_string(v.root_subscriber_fid))
            .release();
    }
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncAddUnsubscribeTask(UnsubscribeTask task,
                                                     OnUnsubscribeTaskReceive hook) const {
    const auto q = queryRepository().template query<query::AddUnsubscribeTask>(
        query::TaskRequestId(task.taskRequestId()),
        query::OwnerUserId(task.ownerUid()),
        query::OwnerFolderIds(task.ownerFids()),
        query::UserId(task.subscriberUid()),
        query::FolderId(task.rootSubscriberFid()));
    db()->fetch(q, wrapHook<reflection::UnsubscribeTask>(std::move(hook), taskFromReflection));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncGetUnsubscribeTasks(std::size_t limit,
                                                      std::chrono::seconds aliveTimeout,
                                                      OnUnsubscribeTaskVecReceive hook) const {
    const auto q = queryRepository().template query<query::GetUnsubscribeTasks>(
        query::SubscriptionsLimit(limit),
        query::AliveTimeoutSeconds(aliveTimeout));
    db()->fetch(q, wrapHook<reflection::UnsubscribeTask>(std::move(hook), taskFromReflection));
}

template <typename DatabaseGenerator>
void SubscriptionRepository<DatabaseGenerator>::asyncRemoveSunscriptionsAndTask(UnsubscribeTask task,
                                                             OnSubscriptionsChunkReceive hook) const {
    const auto q = queryRepository().template query<query::RemoveSubscriptionsAndTask>(
        query::TaskId(task.taskId()));
    db()->fetch(q, wrapHook<reflection::Subscription>(std::move(hook), detail::subscriptionFromReflection));
}

} // namespace pg
} // namespace macs
