#pragma once

#include <macs/hooks.h>
#include <macs/tabs_repository.h>
#include <macs/errors.h>

#include <pgg/database/database.h>
#include <internal/not_implemented.h>
#include <pgg/query/repository.h>
#include <internal/query/ids.h>
#include <internal/query/comment.h>
#include <macs_pg/logging.h>

namespace macs {
namespace pg {

typedef query::UserId UID;

template <typename DatabaseGenerator>
class TabsRepository : public macs::TabsRepository {
public:
    typedef macs::TabsRepository Base;

    TabsRepository(const DatabaseGenerator& generator,
                   const std::string & uid,
                   pgg::query::RepositoryPtr queryRepository,
                   const pgg::RequestInfo& requestInfo,
                   logging::v2::LogPtr logger)
        : Base()
        , uid_(uid)
        , queryRepository_(queryRepository)
        , db(generator)
        , requestInfo_(requestInfo)
        , logger_(logger) {
    }

    virtual ~TabsRepository() {}

private:

    virtual void asyncGetTabs(OnTabs) const override;
    virtual void asyncGetOrCreateTab(const Tab::Type&, OnUpdateTab) const override;
    virtual void asyncResetFresh(const Tab::Type&, OnUpdate) const override;
    virtual void asyncCanReadTabs(Hook<bool>) const override;

private:
    const std::string & uid() const {
        return uid_;
    }
    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }
    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    template <typename T, typename ...ArgsT >
    T queryUpdate(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                requestInfo_, std::forward<ArgsT>(args) ...);
    }
    std::shared_ptr<const TabsRepository> getSelf() const {
        return std::dynamic_pointer_cast<const TabsRepository>(pgg::share(this));
    }

    const std::string uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
    const pgg::RequestInfo requestInfo_;
    logging::v2::LogPtr logger_;
};

template <typename DatabaseGenerator>
TabsRepositoryPtr createTabsRepository(
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo,
    logging::v2::LogPtr logger);

} // namespace pg
} // namespace macs

#include "repository.ipp"
