#include <internal/tab/repository.h>
#include <internal/tab/query.h>
#include <internal/tab/factory.h>
#include <internal/tab/converter.h>

#include <pgg/cast.h>
#include <internal/hooks/wrap.h>
#include <internal/reflection/tab.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
void TabsRepository<DatabaseGenerator>::asyncGetTabs(OnTabs hook) const {
    db()->fetch(query<query::AllTabsList>(),
                [log = logger_, h = std::move(hook)](error_code e, auto data) {
        if(!e) {
            std::vector<reflection::Tab> ts;
            boost::transform(data, std::back_inserter(ts), [] (auto row) {
                return pgg::cast<reflection::Tab>(row);
            });
            h(makeTabSet(std::move(ts), log, "FoldersRepository::syncGetFolders"));
        } else {
            h(std::move(e));
        }
    });
}

template <typename DatabaseGenerator>
void TabsRepository<DatabaseGenerator>::asyncGetOrCreateTab(const Tab::Type& type,
                                         OnUpdateTab hook) const {
    const auto q = queryUpdate<query::GetOrCreateTab>(query::TabType(type));
    db()->fetch(q, wrapHook<reflection::Tab>(std::move(hook),[](auto v) {
        return macs::pg::TabFactory().fromReflection(v).release();
    }));
}

template <typename DatabaseGenerator>
void TabsRepository<DatabaseGenerator>::asyncResetFresh(const Tab::Type& type,
                                     OnUpdate hook) const {
    const auto q = queryUpdate<query::ResetTabFresh>(query::TabType(type));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void TabsRepository<DatabaseGenerator>::asyncCanReadTabs(Hook<bool> hook) const {
    const auto q = query<query::CanReadTabs>();
    db()->fetch(q, wrapHook<reflection::CanReadTabs>(std::move(hook),[](auto v) {
        return v.can_read_tabs;
    }));
}

} // namespace pg
} // namespace macs
