#ifndef INCLUDE_INTERNAL_THREAD_ADD_MESSAGE_TO_THREAD_H_
#define INCLUDE_INTERNAL_THREAD_ADD_MESSAGE_TO_THREAD_H_

#include <pgg/query/transactional.h>
#include <internal/envelope/query.h>
#include <internal/thread/query.h>
#include <internal/thread/threads_chooser.h>
#include <internal/thread/join_thread_result.h>
#include <internal/thread/mute_message_if_thread_muted.h>
#include <internal/thread/mark_answered_messages.h>
#include <pgg/query/ranges.h>
#include <pgg/cast.h>
#include <pgg/range.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/store_message.h>
#include <internal/reflection/find_threads.h>
#include <internal/query/comment.h>

#include <iterator>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace macs{
namespace pg{

using PGEnvelope = macs::pg::query::Envelope;

template <typename TransactionPtr>
class AddMessageToThread: public boost::asio::coroutine,
                          public boost::enable_shared_from_this<AddMessageToThread<TransactionPtr>>{
public:
    using Handler = std::function<void(error_code, const JoinThreadResult&)>;

    explicit AddMessageToThread (TransactionPtr t, pgg::query::RepositoryPtr qr, const std::string& uid,
            const pgg::RequestInfo& ri, ThreadMeta tm, const RfcMessageId& messageId, const LabelSet& allLabels,
            const FolderSet& folders, const std::vector<Lid>& envelopeLabels, const Fid& envelopeFid, std::time_t receiveDate,
            const Handler& handler)
        : transaction_(t), queryRepository_(qr), uid_(uid), requestInfo_(ri), meta_(std::move(tm)),
        messageId_(messageId), allLabels_(allLabels), folders_(folders), envelopeLabels_(envelopeLabels), receiveDate_(receiveDate),
        envelopeFid_(envelopeFid), handler_(handler)
    {
    }

    void operator()() {
        const auto cb = [this, self=this->shared_from_this()](auto err){
            if (err) {
                handler_(std::move(err), {});
            } else {
                (*this)();
            }
        };

        reenter (this) {
            if (meta_.mergeRule == ThreadsMergeRules::hash) {
                yield findThreadsByHash();
            } else if (meta_.mergeRule == ThreadsMergeRules::references) {
                for (offset_ = 0; offset_ < meta_.referenceHashes.size() &&
                        foundThreads_.empty(); offset_ += maxRefsCount_) {
                    yield findThreadsByReferences(offset_, maxRefsCount_);
                }
            }

            chooseThreadsForJoin( std::move(foundThreads_) ).swap( filteredThreads_ );

            if (filteredThreads_.size() == 1) {
                result_.tid = filteredThreads_.back();
            } else if (filteredThreads_.size() > 1) {
                yield joinThreads();
            }

            if (!result_.tid.empty()) {
                yield muteMessageIfThreadMuted();

                yield markAnsweredMessages();
            }

            handler_(error_code(), result_);
        }
    }

private:
    void findThreadsByHash() {
        using namespace macs::pg::query;

        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleFindThreads(std::move(err), std::move(data));
        };
        const auto q = query<FindThreadsByHash>( meta_.limits, meta_.hash );
        transaction_->fetch(q, h);
    }

    void findThreadsByReferences(std::size_t offset, std::size_t count) {
        using namespace macs::pg::query;

        using OffsetType = typename std::remove_pointer_t<decltype(meta_.referenceHashes.begin())>::difference_type;
        auto b = std::next(meta_.referenceHashes.begin(), static_cast<OffsetType>(offset));
        auto e = std::next(b, static_cast<OffsetType>(std::min(count, meta_.referenceHashes.size() - offset)));

        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleFindThreads(std::move(err), std::move(data));
        };
        const auto q = query<FindThreadsByReferences>(
                MailIdVec( !messageId_.empty() ? macs::MidVec({ messageId_ }) : macs::MidVec() ),
                MailRefVec( macs::MailRefVec(b, e) ) );
        transaction_->fetch(q, h);
    }

    void joinThreads() {
        using namespace macs::pg::query;
        result_.tid = filteredThreads_.front();

        auto b = std::next(filteredThreads_.begin());
        auto e = filteredThreads_.end();

        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleJoinThreads(std::move(err), std::move(data));
        };
        const auto q = query<JoinThreads>( ThreadIdVector({b, e}), ThreadId( result_.tid ) );
        transaction_->fetch(q, h);
    }

    void muteMessageIfThreadMuted() {
        const auto h = [this, self=this->shared_from_this()](error_code err, const std::vector<Lid>& addLids){
            if (err) {
                handler_(std::move(err), {});
            } else {
                result_.addLids.insert(result_.addLids.end(), addLids.begin(), addLids.end());
                (*this)();
            }
        };
        auto coro = boost::make_shared<MuteMessageIfThreadMuted<TransactionPtr>>(transaction_, queryRepository_,
            uid_, allLabels_, result_.tid, h);
        (*coro)();
    }

    void markAnsweredMessages() {
        const auto h = [this, self=this->shared_from_this()](error_code err, const std::vector<Lid>& removeLids){
            if (err) {
                handler_(std::move(err), {});
            } else {
                result_.removeLids.insert(result_.removeLids.end(), removeLids.begin(), removeLids.end());
                (*this)();
            }
        };
        auto coro = boost::make_shared<MarkAnsweredMessages<TransactionPtr>>(transaction_, queryRepository_,
            uid_, allLabels_, folders_, result_.tid, envelopeLabels_, receiveDate_, envelopeFid_, h);
        (*coro)();
    }

    template <typename DataRange>
    void handleFindThreads(error_code err, DataRange data) {
        if (err) {
            handler_(std::move(err), {});
        } else {
            boost::transform(data, std::back_inserter( foundThreads_ ),
                [] (const auto& r) {
                auto v = pgg::cast< reflection::FindThreads >(r);
                return ThreadInfo{ std::to_string( v.tid ), v.msg_count }; });
            (*this)();
        }
    }

    template <typename DataRange>
    void handleJoinThreads(error_code err, DataRange data) {
        if (err) {
            handler_(std::move(err), {});
        } else if (data.empty()) {
            auto err = error_code(pgg::error::noDataReceived,
                "No data received as a result of JoinThreads" );
            handler_(std::move(err), {});
        } else {
            (*this)();
        }
    }

    TransactionPtr transaction_;
    pgg::query::RepositoryPtr queryRepository_;
    std::string uid_;
    pgg::RequestInfo requestInfo_;

    ThreadMeta meta_;
    ThreadInfoVec foundThreads_;
    TidVector filteredThreads_;
    RfcMessageId messageId_;
    JoinThreadResult result_;
    LabelSet allLabels_;
    macs::FolderSet folders_;
    std::vector<Lid> envelopeLabels_;
    std::time_t receiveDate_ = 0;
    Fid envelopeFid_;
    Handler handler_;

    std::size_t maxRefsCount_ = 100;
    std::size_t offset_ = 0;

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository_, uid_, std::forward<Args>(args)...);
    }
};

}   //namespace pg
}   //namespace macs

#include <boost/asio/unyield.hpp>

#endif /* INCLUDE_INTERNAL_THREAD_ADD_MESSAGE_TO_THREAD_H_ */
