#pragma once

#include <internal/query/comment.h>
#include <internal/reflection/mid_and_received_date.h>
#include <internal/reflection/mid.h>

#include <pgg/query/transactional.h>
#include <pgg/query/ranges.h>
#include <pgg/cast.h>
#include <pgg/range.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

using FSymbol = macs::Folder::Symbol;

template <typename TransactionPtr>
class MarkAnsweredMessages: public boost::asio::coroutine,
                            public boost::enable_shared_from_this<MarkAnsweredMessages<TransactionPtr>>
{
public:
    using Handler = std::function<void(error_code, const std::vector<Lid>& removeLids)>;

    MarkAnsweredMessages(TransactionPtr t, pgg::query::RepositoryPtr qr, const std::string& uid,
            const LabelSet& allLabels, const FolderSet& folders, const ThreadId& tid,
            const std::vector<Lid>& envelopeLabels, std::time_t receiveDate, const Fid& envelopeFid, const Handler& handler)
        : transaction_(t), queryRepository_(qr), uid_(uid), allLabels_(allLabels), tid_(tid), handler_(handler),
        envelopeLabels_(envelopeLabels.begin(), envelopeLabels.end()), receiveDate_(receiveDate), envelopeFid_(envelopeFid)
    {
        static const std::vector<FSymbol> outgoingSymbols = { FSymbol::drafts, FSymbol::sent, FSymbol::outbox };
        for (const auto& symbol: outgoingSymbols) {
            if (folders.exists(symbol)) {
                outgoingFolders_.emplace_back(folders.fid(symbol));
            }
        }
    }

    void operator()() {
        reenter (this) {
            if (allLabels_.exists(Label::Symbol::remindNoAnswer_label)) {
                remindNoAnswerLid_ = allLabels_.at(Label::Symbol::remindNoAnswer_label).lid();

                if (envelopeLabels_.count(remindNoAnswerLid_)) {
                    yield getNewestMessageInThread();
                    if (newestMessageInThreadDate_ > receiveDate_) {
                        removeLids_.push_back(remindNoAnswerLid_);
                    }
                }

                if (!boost::count(outgoingFolders_, envelopeFid_)) {
                    yield getNotAnsweredMids();
                    if (!notAnsweredMids_.empty()) {
                        yield unmark(notAnsweredMids_, remindNoAnswerLid_);
                    }
                }
            }

            handler_(error_code(), removeLids_);
        }
    }

private:
    void getNewestMessageInThread() {
        using namespace macs::pg::query;

        const auto h = [this, self=this->shared_from_this()](auto err, auto data) {
            this->handleGetNewestMessage(std::move(err), std::move(data));
        };
        const auto q = query<GetNewestMessageInThread>(ThreadId(tid_),
            OutgoingFolders(outgoingFolders_));
        transaction_->fetch(q, h);
    }

    void getNotAnsweredMids() {
        using namespace macs::pg::query;

        const auto h = [this, self=this->shared_from_this()](auto err, auto data) {
            this->handleGetNotAnsweredMids(std::move(err), std::move(data));
        };
        const auto q = query<GetNotAnsweredMids>(ThreadId(tid_), RemindNoAnswerLid(remindNoAnswerLid_), ReceivedDate(receiveDate_));
        transaction_->fetch(q, h);
    }

    void unmark(const std::list<Mid>& mids, const Lid& lid) {
        using namespace macs::pg::query;
        const auto h = [this, self=this->shared_from_this()](auto err) {
            this->handleUnmark(std::move(err));
        };
        const auto q = query<RemoveLabels>(MailIdList(mids), LabelIdList({lid}));
        transaction_->execute(q, h);
    }

    template <typename DataRange>
    void handleGetNewestMessage(error_code err, DataRange data) {
        if (err) {
            return handler_(std::move(err), {});
        }
        if (!data.empty()) {
            auto v = pgg::cast< reflection::MidAndReceivedDate >(data.front());
            newestMessageInThreadDate_ = v.received_date;
        }
        (*this)();
    }

    template <typename DataRange>
    void handleGetNotAnsweredMids(error_code err, DataRange data) {
        if (err) {
            return handler_(std::move(err), {});
        }
        for (const auto& row: data) {
            auto v = pgg::cast< reflection::Mid >(row);
            notAnsweredMids_.push_back(std::to_string(v.mid));
        }
        (*this)();
    }

    void handleUnmark(error_code err) {
        if (err) {
            handler_(std::move(err), {});
        } else {
            (*this)();
        }
    }

    TransactionPtr transaction_;
    pgg::query::RepositoryPtr queryRepository_;
    std::string uid_;
    LabelSet allLabels_;
    ThreadId tid_;
    Handler handler_;
    std::vector<Lid> removeLids_;
    std::vector<Fid> outgoingFolders_;
    std::set<Lid> envelopeLabels_;
    std::time_t receiveDate_ = 0;
    Fid envelopeFid_;
    Lid remindNoAnswerLid_;
    std::time_t newestMessageInThreadDate_ = 0;
    std::list<Mid> notAnsweredMids_;

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository_, uid_, std::forward<Args>(args)...);
    }
};


} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
