#pragma once

#include <internal/reflection/thread_labels_list.h>
#include <internal/query/comment.h>

#include <pgg/query/transactional.h>
#include <pgg/query/ranges.h>
#include <pgg/cast.h>
#include <pgg/range.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

template <typename TransactionPtr>
class MuteMessageIfThreadMuted: public boost::asio::coroutine,
                                public boost::enable_shared_from_this<MuteMessageIfThreadMuted<TransactionPtr>>
{
public:
    using Handler = std::function<void(error_code, const std::vector<Lid>& addLids)>;

    MuteMessageIfThreadMuted(TransactionPtr t, pgg::query::RepositoryPtr qr, const std::string& uid,
            const LabelSet& allLabels, const ThreadId& tid, const Handler& handler)
        : transaction_(t), queryRepository_(qr), uid_(uid), allLabels_(allLabels), tid_(tid), handler_(handler)
    {
    }

    void operator()() {
        reenter (this) {
            if (allLabels_.exists(Label::Symbol::mute_label)) {
                muteLid_ = allLabels_.at(Label::Symbol::mute_label).lid();

                yield getThreadLabels();
                if (threadLids_.count(muteLid_)) {
                    addLids_.push_back(muteLid_);
                    if (allLabels_.exists(Label::Symbol::seen_label)) {
                        addLids_.push_back(allLabels_.at(Label::Symbol::seen_label).lid());
                    }
                }
            }
            handler_(error_code(), addLids_);
        }
    }

private:
    void getThreadLabels() {
        using namespace macs::pg::query;

        const auto h = [this, self=this->shared_from_this()](auto err, auto data) {
            this->handleGetThreadLabels(std::move(err), std::move(data));
        };
        const auto q = query<ThreadLabels>(ThreadIdVector({ tid_ }));
        transaction_->fetch(q, h);
    }

    template <typename DataRange>
    void handleGetThreadLabels(error_code err, DataRange data) {
        if (err) {
            return handler_(std::move(err), {});
        }
        if (!data.empty()) {
            const auto& row = data.front();
            auto v = pgg::cast< reflection::ThreadLabelsList >(row);
            boost::transform(v.labels, std::inserter(threadLids_, threadLids_.begin()),
                [] (const auto& l) {
                return std::to_string(l.lid); });
        }
        (*this)();
    }

    TransactionPtr transaction_;
    pgg::query::RepositoryPtr queryRepository_;
    std::string uid_;
    LabelSet allLabels_;
    ThreadId tid_;
    Handler handler_;
    Lid muteLid_;
    std::set<Lid> threadLids_;
    std::vector<Lid> addLids_;

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository_, uid_, std::forward<Args>(args)...);
    }
};


} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
