#pragma once

#include <functional>
#include <macs/hooks.h>
#include <macs/labels_repository.h>
#include <macs/threads_repository.h>

#include <internal/not_implemented.h>
#include <pgg/query/repository.h>
#include <internal/query/ids.h>
#include <internal/query/comment.h>

namespace macs {
namespace pg {

using UID = query::UserId;

template <typename DatabaseGenerator>
struct ThreadsRepository : public macs::ThreadsRepository {
    ThreadsRepository(const DatabaseGenerator& generator, LabelsRepositoryPtr labels,
            const std::string & uid, pgg::query::RepositoryPtr queryRepository, const pgg::RequestInfo& requestInfo)
        : uid_(uid), labels_(labels), queryRepository_(queryRepository), db(generator),
          requestInfo_(requestInfo)
    { }

protected:
    void syncThreadsParticipantsList(const TidVector& tids,
                                     OnParticipantsReceive handler) const override;

    void syncThreadLabelsList(const TidVector& tids, OnThreadLabelsReceive handler) const override;

    void syncFillIdsMap(const MidList& mids, const TidList& tids, const Lids& lidsToCheck,
                        OnThreadMailboxItemsReceive handler) const override;

    void syncFillIdsList(const MidList& mids,
                         const TidList& tids,
                         OnMidsReceive handler) const override;

    void syncFillIdsListWithoutStatus(const MidList& mids,
                                      const TidList& tids,
                                      Envelope::Status excludeStatus,
                                      OnMidsReceive handler) const override;


    void syncFillIdsFList(const TidList& tidsf,
                          const Fid& fid,
                          OnMidsReceive) const override;
    void syncFindThreadsByHash(const ThreadHash & hash,
            const ThreadLimits & limits,
            OnThreadIdsReceive handler) const override;

    void syncFindThreadsByReferences(const MidVec & mids,
            const MailRefVec & refs,
            OnThreadIdsReceive handler) const override;

    void syncJoinThreads( const ThreadId & tid, const TidVector & joinTids,
            OnThreadIdReceive handler) const override;

    void syncFilterThreadsByLid (const TidVector & /*tids*/, const std::string & /*lid*/,
            OnThreadIdsReceive /*handler*/) const override NOT_IMPLEMENTED

    void syncMessageCountInThreads(const TidVector& tids,
        OnCountReceive handler) const override;

private:
    const macs::LabelsRepository & labels() const {
        return *labels_;
    }

    auto getSelf() const {
        return std::dynamic_pointer_cast<const ThreadsRepository>(pgg::share(this));
    }

    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }
    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid_,
                std::forward<ArgsT>(args) ...);
    }

    template <typename T, typename ...ArgsT >
    T queryUpdate(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid_,
                pgg::RequestInfo(requestInfo_), std::forward<ArgsT>(args) ...);
    }

    std::string uid_;
    LabelsRepositoryPtr labels_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
    const pgg::RequestInfo requestInfo_;
};

template <typename DatabaseGenerator>
ThreadsRepositoryPtr createThreadsRepository(
    DatabaseGenerator dbg,
    LabelsRepositoryPtr labelsRepository,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo);

} // namespace macs
} // namespace pg

#include "repository.ipp"
