#include <internal/thread/repository.h>
#include <internal/thread/query.h>
#include <pgg/query/ranges.h>
#include <internal/envelope/status_to_label.h>
#include <internal/reflection/find_threads.h>
#include <internal/reflection/operation_result.h>
#include <internal/reflection/thread_mailbox_item.h>
#include <internal/reflection/thread_labels_list.h>
#include <internal/reflection/count.h>
#include <internal/thread/structures/participant_grouper.h>
#include <internal/envelope/factory.h>
#include <internal/hooks/wrap.h>
#include <boost/range.hpp>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncThreadsParticipantsList(const TidVector& tids,
                                                    OnParticipantsReceive handler) const {
    auto q = query<ParticipantsList>(query::ThreadIdVector(tids));
    db()->fetch(q, wrapHook<reflection::ParticipantWithTid>(std::move(handler),
            [](auto v){return std::move(v);},
            [](auto v){return ParticipantGrouper().groupByTid(v);}));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncThreadLabelsList(const TidVector& tids,
                                             OnThreadLabelsReceive handler) const {

    auto q = query<ThreadLabels>(query::ThreadIdVector(tids));
    db()->fetch(q, wrapHook<reflection::ThreadLabelsList>(std::move(handler),[](auto v){
        ThreadLabelsFactory f;
        f.setTid(std::to_string(v.tid));
        f.setRevision(PGG_NUMERIC_CAST(Revision, v.revision));
        for (const auto& x : v.labels) {
            f.addLabel(std::to_string(x.lid), PGG_NUMERIC_CAST(unsigned, x.message_count));
        }
        return f.product();
    }));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFillIdsMap(const MidList& mids, const TidList& tids, const Lids& lidsToCheck,
                                       OnThreadMailboxItemsReceive handler) const {
    auto q = query<MailsInThreads>(query::ThreadIdList(tids), query::MailIdList(mids),
                                   query::LabelIdToCheckList(lidsToCheck));
    db()->fetch(q, wrapHook<reflection::ThreadMailboxItem>(std::move(handler), [](auto v) {
        ThreadMailboxItem item;

        item.seen = v.seen;
        item.spam = v.spam;
        item.mid = std::to_string(v.mid);
        item.fid = std::to_string(v.fid);
        item.receivedDate = std::to_string(v.received_date);
        item.stid = std::move(v.st_id);
        if (v.tid) {
            item.threadId = std::to_string(v.tid.get());
        }
        if (v.tab) {
            item.tab = Tab::Type::fromString(v.tab.get(), std::nothrow);
        }
        std::transform(v.lids_to_check.begin(), v.lids_to_check.end(),
                       std::back_inserter(item.lidsToCheck), [](std::int32_t lid) {
            return std::to_string(lid);
        });
        return item;
    }));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFillIdsList(const MidList& mids,
                                        const TidList& tids,
                                        OnMidsReceive handler) const {

    syncFillIdsMap(mids, tids, {}, [handler](error_code e, auto data) {
        handler(std::move(e), getMids(data));
    });
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFillIdsListWithoutStatus(const MidList& mids,
                                                     const TidList& tids,
                                                     Envelope::Status excludeStatus,
                                                     OnMidsReceive hook) const {
    switch (excludeStatus) {
        case Envelope::Status_read:
        case Envelope::Status_unread: {
            const auto q = query<MidsInThreadsWithSeen>(query::ThreadIdList(tids), query::MailIdList(mids), seenFromStatus(excludeStatus));
            db()->fetch(q, wrapHook(std::move(hook)));
        } break;
        case Envelope::Status_forwarded:
        case Envelope::Status_replied: {
            const auto& labelsSet = labels().getAllLabels();
            const Label excludeLabel = statusToLabel(labelsSet, excludeStatus);
            db()->fetch(query<MidsInThreadsWithoutLabel>(query::ThreadIdList(tids), query::MailIdList(mids), query::LabelId(excludeLabel.lid())), wrapHook(std::move(hook)));
        } break;
    }
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFillIdsFList(const TidList &tidsf,
                                         const Fid& fid,
                                         OnMidsReceive handler) const {
    db()->fetch(query<MidsInFolder>(query::ThreadIdList(tidsf), query::FolderId(fid)), wrapHook(handler));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFindThreadsByHash(const ThreadHash & hash,
            const ThreadLimits & limits,
            OnThreadIdsReceive handler) const {
    db()->fetch(query<FindThreadsByHash>(limits, hash), wrapHook(std::move(handler)));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncFindThreadsByReferences(const MidVec & mids,
            const MailRefVec & refs,
            OnThreadIdsReceive handler) const {
    db()->fetch(query<FindThreadsByReferences>(query::MailIdVec(mids), query::MailRefVec(refs)),
            wrapHook(std::move(handler)));
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncJoinThreads( const ThreadId & tid,
            const TidVector & joinTids,
            OnThreadIdReceive handler) const {
    db()->fetch(query<JoinThreads>(query::ThreadIdVector(joinTids), query::ThreadId(tid)),
            [handler = std::move(handler), tid](error_code err, auto) {
                handler(std::move(err), tid);
    });
}

template <typename DatabaseGenerator>
void ThreadsRepository<DatabaseGenerator>::syncMessageCountInThreads(const TidVector& tids,
                                                  OnCountReceive handler) const {
    auto hook = wrapHook(
        std::move(handler),
        [](std::ostream& s) { s << "syncMessageCountInThreads()"; }
    );

    db()->fetch(query<MessageCountInThreads>(query::ThreadIdVector(tids)), hook);
}

} // namespace pg
} // namespace macs
