#include <internal/hooks/wrap.h>
#include <internal/query/ids.h>
#include <internal/reflection/userinfo.h>
#include <internal/reflection/archive.h>
#include <internal/reflection/backup.h>
#include <internal/user/query.h>
#include <internal/user/state.h>
#include <macs/archive_error.h>
#include <pgg/cast.h>

namespace macs::pg {

namespace ydr = yamail::data::reflection;
namespace {

template<class Result>
decltype(auto) onRes(OnExecute handler) {
    using OptStr = boost::optional<std::string>;
    Hook<OptStr> hook = [h = std::move(handler)] (error_code ec, const auto& result) {
        if (ec) {
            h(ec);
        } else if (result) {
            h(make_error(ydr::from_string<Result>(*result)));
        } else {
            h(error_code());
        }
    };

    return wrapHook<reflection::WithResult>(
        std::move(hook),
        [] (reflection::WithResult data) {
            return data.result;
        }
    );
}
}


template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncFreezingInfo(OnFreezingInfo handler) const {
    const auto q = query<query::FreezingInfo>();
    auto factory = [](reflection::FreezingInfo data) -> FreezingInfo {
        return {macs::pg::UserState::fromString(data.state), data.last_state_update, data.notifies_count};
    };
    db()->fetch(q, wrapHook<reflection::FreezingInfo>(std::move(handler), factory));
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncChangeUserState(macs::UserState fromState, macs::UserState toState, OnExecute handler) const {
    const auto q = queryUpdate<query::ChangeUserState>(query::FromUserState(fromState), query::ToUserState(toState));
    db()->execute(q, std::move(handler));
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncGetArchive(onArchive handler) const {
    const auto q = query<query::GetArchive>();

    const auto factory = [](reflection::OptionalArchive optArchive) -> std::optional<Archive> {
        if (!optArchive.archive) {
            return {};
        }
        
        const auto toStd = []<typename T>(boost::optional<T>& opt) -> std::optional<T> {
            return opt ? std::make_optional(std::move(opt.value())) : std::nullopt;
        };

        reflection::Archive& data = *optArchive.archive;
        return Archive{
                .version=std::move(data.version),
                .state=ydr::from_string<::macs::ArchiveState>(data.state),

                .message_count=std::move(data.message_count),
                .restored_message_count=std::move(data.restored_message_count),

                .updated=std::move(data.updated),
                .notice=toStd(data.notice),
                .task_id=toStd(data.task_id)
        };
    };
    
    db()->fetch(q, wrapHook<reflection::OptionalArchive>(std::move(handler), factory));
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncActivateUserAndUpdateArchiveState(macs::ArchiveState fromState, macs::ArchiveState toState, OnExecute handler) const {
    db()->fetch(
        query<query::ActivateUserAndUpdateArchiveState>(query::FromArchiveState(fromState), query::ToArchiveState(toState)),
        onRes<ExchangeArchiveStateStatus>(std::move(handler))
    );
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncExchangeArchiveTaskId(macs::TaskId from, macs::TaskId to, OnExecute handler) const {
    const auto q = query<query::ExchangeArchiveTaskId>(query::FromTaskId(from), query::ToTaskId(to));
    db()->execute(q, std::move(handler));
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncExchangeArchiveState(macs::UserState userState, macs::ArchiveState from, macs::ArchiveState to,
                                        std::optional<std::string> notice, OnExecute handler) const {
    db()->fetch(
        query<query::ExchangeArchiveState>(
            query::FromUserState(userState), 
            query::FromArchiveState(from), query::ToArchiveState(to),
            query::OptionalNotice(notice)
            ),
        onRes<ExchangeArchiveStateStatus>(std::move(handler))
    );
}

template <typename DatabaseGenerator>
void UsersRepository<DatabaseGenerator>::asyncUpdateRestorationProgress(TaskId fromTaskId, std::int32_t restoredMessageCount, OnExecute handler) const {
    const auto q = query<query::UpdateRestorationProgress>(
        query::TaskId(fromTaskId), query::RestoredMessageCount(restoredMessageCount));
    db()->execute(q, std::move(handler));
}


}
