#ifndef MACS_CHANGE_QUEUE_REPOSITORY_H_28032017
#define MACS_CHANGE_QUEUE_REPOSITORY_H_28032017

#include <macs/io.h>
#include <macs/hooks.h>
#include <macs_pg/subscription/types.h>
#include <macs_pg/changelog/types.h>

namespace macs {

typedef std::vector<ChangeId> ChangeIdChunk;
typedef Hook<ChangeIdChunk> OnChangeIdChunkReceive;
typedef Hook<std::vector<ChangeReference>> OnChangeReferenceChunk;

class ChangeQueueRepository : public std::enable_shared_from_this<ChangeQueueRepository> {
public:
    virtual ~ChangeQueueRepository() = default;

    template <typename Handler = io::sync_context>
    auto get(std::string uid,
             SubscriptionId subscriptionId,
             size_t count,
             Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangeIdChunkReceive> init(handler);
        asyncGet(std::move(uid), subscriptionId, count, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto get(WorkerId id, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangeReferenceChunk> init(handler);
        asyncGet(std::move(id), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto remove(std::vector<ChangeReference> changes,
                Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncRemove(std::move(changes), init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncGet(std::string uid,
                          SubscriptionId subscriptionId,
                          size_t count,
                          OnChangeIdChunkReceive hook) const = 0;

    virtual void asyncGet(WorkerId id,
            OnChangeReferenceChunk hook) const = 0;

    virtual void asyncRemove(std::vector<ChangeReference> changes,
                             OnExecute hook) const = 0;

};


typedef std::shared_ptr<ChangeQueueRepository> ChangeQueueRepositoryPtr;

} // namespace macs

#endif // MACS_CHANGE_QUEUE_REPOSITORY_H_28032017
