#pragma once

#include <boost/optional.hpp>

#include <macs/revision.h>
#include <macs/hooks.h>
#include <macs_pg/changelog/change_type.h>
#include <macs_pg/changelog/types.h>

namespace macs {

struct ChangeData {
    using Type = macs::pg::ChangeType;
    ChangeId changeId;
    std::string uid;
    Revision revision;
    Type type = Type::unknown;
    boost::optional<std::string> changed;
    boost::optional<std::string> arguments;

    static const ChangeData default_;
};


template <typename Impl>
class ChangeDataInterface
{
public:
#define GETTER(NAME) auto& NAME() const { return data().NAME; }
    GETTER(changeId)
    GETTER(uid)
    GETTER(revision)
    GETTER(type)
    GETTER(changed)
    GETTER(arguments)
#undef GETTER
private:
    friend Impl;
    ChangeDataInterface() = default;
    ChangeDataInterface(const ChangeDataInterface&) = default;
    ChangeDataInterface(ChangeDataInterface&&) = default;
    ChangeDataInterface& operator = (const ChangeDataInterface&) = default;
    ChangeDataInterface& operator = (ChangeDataInterface&&) = default;

    const ChangeData& data() const {
        return static_cast<const Impl&>(*this).Impl::data();
    }
};


class Change: public ChangeDataInterface<Change> {
public:
    Change() = default;
    Change(std::shared_ptr<const ChangeData> data): data_(std::move(data))
        {}
    Change(const Change&) = default;
    Change(Change&&) = default;
    Change& operator=(const Change&) = default;
    Change& operator=(Change&&) = default;
protected:
    friend class ChangeFactory;
    friend class ChangeDataInterface<Change>;
    const ChangeData& data() const {
        return data_ ? *data_ : ChangeData::default_;
    }
    std::shared_ptr<const ChangeData> data_;
};


typedef Hook<std::vector<Change>> OnChangesReceive;

typedef Hook<ChangeId> OnChangeIdReceive;

typedef Hook<Optional<ChangeId>> OnChangesIdReceive;

} // namespace macs
