#pragma once

#include <boost/optional.hpp>
#include <macs/hooks.h>
#include <macs/io.h>

#include <macs_pg/changelog/change.h>
#include <macs_pg/changelog/change_type.h>

namespace macs {


class ChangeLogRepository : public std::enable_shared_from_this<ChangeLogRepository> {
public:
    virtual ~ChangeLogRepository() = default;

    template <typename Handler = io::sync_context>
    auto getChanges(std::vector<ChangeId> ids,
                   Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangesReceive> init(handler);
        asyncGetChanges(std::move(ids), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMinimumKnownChangeId(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangeIdReceive> init(handler);
        asyncGetMimimumKnownChangeId(init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncGetChanges(std::vector<ChangeId>, OnChangesReceive hook) const = 0;
    virtual void asyncGetMimimumKnownChangeId(OnChangeIdReceive hook) const = 0;
};


typedef std::shared_ptr<ChangeLogRepository> ChangeLogRepositoryPtr;

class UserChangeLogRepository : public std::enable_shared_from_this<UserChangeLogRepository> {
public:
    virtual ~UserChangeLogRepository() = default;

    /**
     * Provides no more than limit changes for with revision greater than specified
     * in DESCENDING order. That means: if total amount of changes exceed limit you
     * will get the lowest change revision greater than specified. If amount of
     * changes feets revision, you will get the lowest revision exactly as specified.
     * If last change has revision as "start" - changes feets in limit, if greater -
     * changes in changelog more than limit.
     */
    template <typename Handler = io::sync_context>
    auto getChanges(Revision start, std::int64_t limit, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangesReceive> init(handler);
        asyncGetChanges(start, limit, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getChangeIdByRevision(Revision revision, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangesIdReceive> init(handler);
        asyncGetChangeIdByRevision(revision, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getChangelog(Revision lastRevision, std::int64_t limit,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangesReceive> init(handler);
        asyncGetChangelog(lastRevision, limit, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getChangelogByType(Revision lastRevision, std::int64_t limit,
                          const std::vector<macs::pg::ChangeType>& changeTypes,
                          Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnChangesReceive> init(handler);
        asyncGetChangelogByType(lastRevision, limit, changeTypes, init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncGetChanges(Revision, std::int64_t, OnChangesReceive hook) const = 0;
    virtual void asyncGetChangeIdByRevision(Revision, OnChangesIdReceive hook) const = 0;
    virtual void asyncGetChangelog(Revision, std::int64_t,
                                   OnChangesReceive hook) const = 0;
    virtual void asyncGetChangelogByType(Revision, std::int64_t,
                                         const std::vector<macs::pg::ChangeType>&,
                                         OnChangesReceive hook) const = 0;
};


typedef std::shared_ptr<UserChangeLogRepository> UserChangeLogRepositoryPtr;

} // namespace macs
