#pragma once

#include <macs/types.h>
#include <macs_pg/collectors/collector_id.h>

#include <boost/serialization/strong_typedef.hpp>

#include <ctime>
#include <memory>
#include <string>
#include <vector>

namespace macs {

BOOST_STRONG_TYPEDEF(uint64_t, OldPopid);

enum class CollectorState {
    undefined,
    created,
    ready,
    disabled,
    deleted,
    migrated,
    unmigrated
};

struct CollectorData {
    Uid uid;
    CollectorId collectorId;
    Uid srcUid;
    std::string authToken;
    Fid rootFolderId;
    Lid labelId;
    bool ignoreFoldersStruct;
    std::time_t creationTs = 0;
    Mid lastMid;
    MidVec skippedMids;
    CollectorState state;
    OldPopid oldPopid;
    Mid migratedLastMid;
    std::string originalServer;
    CollectorState migrationTargetState = CollectorState::undefined;

    static const CollectorData default_;
};

template <typename Impl>
class CollectorDataInterface {
public:
#define GETTER(NAME) auto& NAME() const { return data().NAME; }
    GETTER(uid)
    GETTER(collectorId)
    GETTER(srcUid)
    GETTER(authToken)
    GETTER(rootFolderId)
    GETTER(labelId)
    GETTER(ignoreFoldersStruct)
    GETTER(creationTs)
    GETTER(lastMid)
    GETTER(skippedMids)
    GETTER(state)
    GETTER(oldPopid)
    GETTER(migratedLastMid)
    GETTER(originalServer)
    GETTER(migrationTargetState)
#undef GETTER

private:
    friend Impl;
    CollectorDataInterface() = default;
    CollectorDataInterface(const CollectorDataInterface&) = default;
    CollectorDataInterface(CollectorDataInterface&&) = default;
    CollectorDataInterface& operator = (const CollectorDataInterface&) = default;
    CollectorDataInterface& operator = (CollectorDataInterface&&) = default;

    const CollectorData& data() const { return static_cast<const Impl&>(*this).Impl::data(); }
};

class Collector: public CollectorDataInterface<Collector> {
public:
    Collector() = default;
    Collector(std::shared_ptr<const CollectorData> data) : data_(std::move(data)) {}
    Collector(const Collector&) = default;
    Collector(Collector&&) = default;
    Collector& operator=(const Collector&) = default;
    Collector& operator=(Collector&&) = default;



private:
    friend class CollectorDataInterface<Collector>;
    friend class CollectorFactory;
    const CollectorData& data() const {
        return data_ ? *data_ : CollectorData::default_;
    }
    std::shared_ptr<const CollectorData> data_;
};

struct CollectorResult {
    Revision rev;
    Collector collector;
};

using CollectorsList = std::vector<Collector>;

} // namespace macs
