#pragma once

#include <macs_pg/collectors/collector_factory.h>
#include <macs_pg/collectors/hooks.h>
#include <macs/hooks.h>
#include <macs/io.h>
#include <memory>

namespace macs {

class CollectorsRepository: public std::enable_shared_from_this<CollectorsRepository> {
public:
    virtual ~CollectorsRepository() = default;

    template <typename Handler = io::sync_context>
    auto createCollector(CollectorFactory factory,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCollectorResult> init(handler);
        asyncCreateCollector(std::move(factory), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto migrateCollector(CollectorFactory factory,
                          Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCollectorResult> init(handler);
        asyncMigrateCollector(std::move(factory), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetCollector(const CollectorFactory& factory,
                        Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnUpdate> init(handler);
        asyncResetCollector(factory, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteCollector(CollectorId collectorId,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnUpdate> init(handler);
        asyncDeleteCollector(collectorId, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateSkippedMids(CollectorId collectorId,
                           const MidVec& skippedMids,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnExecute> init(handler);
        asyncUpdateSkippedMids(collectorId, skippedMids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateLastMid(CollectorId collectorId,
                       const Mid& lastMid,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnExecute> init(handler);
        asyncUpdateLastMid(collectorId, lastMid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateState(CollectorId collectorId,
                     const CollectorState& newState,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnUpdate> init(handler);
        asyncUpdateState(collectorId, newState, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateMigrationTargetState(CollectorId collectorId,
                                    const CollectorState& newState,
                                    Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnUpdate> init(handler);
        asyncUpdateMigrationTargetState(collectorId, newState, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateCollector(const CollectorFactory& factory,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnUpdate> init(handler);
        asyncUpdateCollector(factory, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetToken(CollectorId collectorId, Handler handler = io::use_sync) const {
        io::detail::init_async_result <Handler, OnExecute> init(handler);
        asyncResetToken(collectorId, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getNextEnvelopeChunk(const Mid& id,
                             const uint64_t count,
                             LabelSet labels,
                             Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnEnvelopeChunkReceive> init(handler);
        asyncGetNextEnvelopeChunk(id, count, std::move(labels), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getNextPop3EnvelopeChunk(const Mid& id,
                                  const uint64_t count,
                                  LabelSet labels,
                                  Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnEnvelopeChunkReceive> init(handler);
        asyncGetNextPop3EnvelopeChunk(id, count, std::move(labels), init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncCreateCollector(CollectorFactory factory,
                                      OnCollectorResult hook) const = 0;
    virtual void asyncMigrateCollector(CollectorFactory factory,
                                       OnCollectorResult hook) const = 0;
    virtual void asyncResetCollector(const CollectorFactory& factory, OnUpdate hook) const = 0;
    virtual void asyncDeleteCollector(CollectorId collectorId, OnUpdate hook) const = 0;
    virtual void asyncUpdateSkippedMids(CollectorId collectorId,
                                        const MidVec& skippedMids,
                                        OnExecute hook) const = 0;
    virtual void asyncUpdateLastMid(CollectorId collectorId,
                                    const Mid& lastMid,
                                    OnExecute hook) const = 0;
    virtual void asyncUpdateState(CollectorId collectorId,
                                  const CollectorState& newState,
                                  OnUpdate hook) const = 0;
    virtual void asyncUpdateMigrationTargetState(CollectorId collectorId,
                                                 const CollectorState& newState,
                                                 OnUpdate hook) const = 0;
    virtual void asyncUpdateCollector(const CollectorFactory& factory, OnUpdate hook) const = 0;
    virtual void asyncResetToken(CollectorId collectorId, OnExecute hook) const = 0;
    virtual void asyncGetNextEnvelopeChunk(const Mid& id,
                                           const uint64_t count,
                                           LabelSet labels,
                                           OnEnvelopeChunkReceive hook) const = 0;
    virtual void asyncGetNextPop3EnvelopeChunk(const Mid& id,
                                               const uint64_t count,
                                               LabelSet labels,
                                               OnEnvelopeChunkReceive hook) const = 0;
};

using CollectorsRepositoryPtr = std::shared_ptr<CollectorsRepository>;

}; // namespace macs
