#pragma once

#include <macs/io.h>
#include <macs/hooks.h>
#include <macs_pg/hooks.h>
#include <macs_pg/collectors/hooks.h>
#include <memory>

namespace macs {

class CollectorsShardRepository: public std::enable_shared_from_this<CollectorsShardRepository> {
public:
    virtual ~CollectorsShardRepository() = default;

    template <typename Handler = io::sync_context>
    auto getUsers(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnShardUserList> init(handler);
        asyncGetUsers(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getChanges(std::time_t lastMovedDate,
                    std::time_t lastDeletedDate,
                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnShardUserList> init(handler);
        asyncGetChanges(lastMovedDate, lastDeletedDate, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getCollectors(std::vector<Uid> uidsList, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCollectorsList> init(handler);
        asyncGetCollectors(uidsList, init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncGetUsers(OnShardUserList hook) const = 0;
    virtual void asyncGetChanges(std::time_t lastMovedDate,
                                 std::time_t lastDeletedDate,
                                 OnShardUserList hook) const = 0;
    virtual void asyncGetCollectors(std::vector<Uid> uidsList, OnCollectorsList hook) const = 0;
};

using CollectorsShardRepositoryPtr = std::shared_ptr<CollectorsShardRepository>;

}
