#pragma once

#include <macs/io.h>
#include <macs/threads_meta.h>
#include <macs/envelopes_repository.h>
#include <macs_pg/mailish/hooks.h>
#include <macs_pg/mailish/entry.h>
#include <macs_pg/mailish/auth_data.h>
#include <macs_pg/mailish/account.h>
#include <macs_pg/mailish/folder.h>
#include <macs_pg/mailish/folder_info.h>
#include <macs_pg/mailish/move_coords.h>
#include <macs_pg/mailish/message_info.h>

namespace macs {

class MailishRepository: public std::enable_shared_from_this<MailishRepository> {
public:
    using MidsList = std::list<std::string>;
    using ImapIds = std::vector<uint64_t>;
    using NotificationMode = EnvelopesRepository::NotificationMode;

    virtual ~MailishRepository() = default;

    template <typename Handler = io::sync_context>
    auto moveMessages(const Fid& srcFid, const Fid& dstFid, const std::optional<Tab::Type>& dstTab, const MailishMoveCoordsChunk& chunk, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(srcFid);
        ASSERT_NOT_EMPTY_ARG(dstFid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncMoveMessages(srcFid, dstFid, dstTab, chunk, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteMailishEntries(const Fid& fid, const ImapIds& imapIds, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDeleteMailishEntries(fid, imapIds, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto incrementErrors(const Fid& fid, const uint64_t imapId, const uint32_t errors, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncIncrementErrors(fid, imapId, errors, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto saveAccount(const MailishAuthData& data, const MailishAccount& account, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncSaveAccount(data, account, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto invalidateAuthData(const std::string& tokenId, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncInvalidateAuthData(tokenId, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getAuthData(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishAuthData> init(handler);
        asyncGetAuthData(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getAccount(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishAccount> init(handler);
        asyncGetAccount(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFolders(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishFolders> init(handler);
        asyncGetFolders(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateDownloadedRange(const Fid& fid, const uint32_t begin, const uint32_t end, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncUpdateDownloadedRange(fid, begin, end, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessages(const Fid& fid, const uint32_t highest_imap_id, const uint32_t count, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnMailishMessages> init(handler);
        asyncGetMessages(fid, highest_imap_id, count, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessages(const Fid& fid, const ImapIds& imap_ids, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnMailishMessages> init(handler);
        asyncGetMessages(fid, imap_ids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessages(const MidVec& mids, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishMessages> init(handler);
        asyncGetMessages(mids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getNotDownloadedMessages(uint32_t count, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishMessages> init(handler);
        asyncGetNotDownloadedMessages(count, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessagesTop(const Fid& fid, const uint32_t count, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnMailishMessages> init(handler);
        asyncGetMessagesTop(fid, count, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto eraseSecurityLocks(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncEraseSecurityLocks(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto initFolder(const Fid& fid, const MailishFolderInfo& info, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(fid);
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncInitFolder(fid, info, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateFolder(const Folder& folder, const MailishFolderInfo& info, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncUpdateFolder(folder, info, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteMailishFolderEntries(const FidVec& fids, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncDeleteMailishFolderEntries(fids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto createFolder(const std::string& name, const Fid& parentId, const Folder::Symbol& symbol, const MailishFolderInfo& info,
        Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMailishFolder> init(handler);
        asyncCreateFolder(name, parentId, symbol, info, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateLastSyncTs(const std::time_t lastSyncTs, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncUpdateLastSyncTs(lastSyncTs, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto store(const Envelope& envelope, const MimeParts& mime, const ThreadMeta& meta, const NotificationMode& notificationMode,
        const MailishMessageInfo& mailishInfo, Handler handler = io::use_sync) const
    {
        io::detail::init_async_result<Handler, OnUpdateEnvelope> init(handler);
        asyncStore(envelope, mime, meta, notificationMode, mailishInfo, init.handler);
        return init.result.get();
    }


protected:
    virtual void asyncMoveMessages(const Fid& srcFid,
                                   const Fid& dstFid,
                                   const std::optional<Tab::Type>& dstTab,
                                   const MailishMoveCoordsChunk& chunk,
                                   OnExecute hook) const = 0;
    virtual void asyncDeleteMailishEntries(const Fid& fid, const ImapIds& imapIds, OnExecute hook) const = 0;
    virtual void asyncIncrementErrors(const Fid& fid, const uint64_t imapId, const uint32_t errors, OnExecute hook) const = 0;
    virtual void asyncSaveAccount(const MailishAuthData& data, const MailishAccount& account, OnExecute hook) const = 0;
    virtual void asyncInvalidateAuthData(const std::string& tokenId, OnExecute hook) const = 0;
    virtual void asyncGetAuthData(OnMailishAuthData hook) const = 0;
    virtual void asyncGetAccount(OnMailishAccount hook) const = 0;
    virtual void asyncGetFolders(OnMailishFolders hook) const = 0;
    virtual void asyncUpdateDownloadedRange(const Fid& fid, const uint32_t begin, const uint32_t end, OnExecute hook) const = 0;
    virtual void asyncGetMessages(const Fid& fid, const uint32_t highest_imap_id, const uint32_t count, OnMailishMessages hook) const = 0;
    virtual void asyncGetMessages(const Fid& fid, const ImapIds& imap_ids, OnMailishMessages hook) const = 0;
    virtual void asyncGetMessages(const MidVec& mids, OnMailishMessages hook) const = 0;
    virtual void asyncGetNotDownloadedMessages(const uint32_t count, OnMailishMessages hook) const = 0;
    virtual void asyncGetMessagesTop(const Fid& fid, const uint32_t count, OnMailishMessages hook) const = 0;
    virtual void asyncEraseSecurityLocks(OnExecute hook) const = 0;
    virtual void asyncInitFolder(const Fid& fid, const MailishFolderInfo& info, OnExecute hook) const = 0;
    virtual void asyncUpdateFolder(const Folder& folder, const MailishFolderInfo& info, OnExecute hook) const = 0;
    virtual void asyncDeleteMailishFolderEntries(const FidVec& fids, OnExecute hook) const = 0;
    virtual void asyncCreateFolder(const std::string& name, const Fid& parentId, const Folder::Symbol& symbol,
        const MailishFolderInfo& info, OnMailishFolder hook) const = 0;
    virtual void asyncUpdateLastSyncTs(const std::time_t lastSyncTs, OnExecute hook) const = 0;
    virtual void asyncStore(const Envelope& envelope, const MimeParts& mime, const ThreadMeta& meta, const NotificationMode& notificationMode,
        const MailishMessageInfo& mailishInfo, OnUpdateEnvelope hook) const = 0;
};

using MailishRepositoryPtr = std::shared_ptr<MailishRepository>;

}; // namespace macs
