#ifndef MACS_SHARED_FOLDERS_REPOSITORY_H
#define MACS_SHARED_FOLDERS_REPOSITORY_H

#include <macs/types.h>
#include <macs/io.h>
#include <macs/hooks.h>

namespace macs {

using OnFidVecReceive = Hook<FidVec>;

class SharedFoldersRepository : public std::enable_shared_from_this<SharedFoldersRepository> {
public:
    virtual ~SharedFoldersRepository() = default;
    template <typename Handler = io::sync_context>
    auto createSharedFolder(Fid fid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncCreateSharedFolder(fid, init.handler);
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto addSubscriber(Fid fid, const std::string& subscriber, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncAddSubscriber(fid, subscriber, init.handler);
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto getAllSharedFolders(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFidVecReceive> init(handler);
        asyncGetAllSharedFolders(init.handler);
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto createSharedFolderWithArchivation(Fid fid,
                                           const Folder::ArchivationType& type,
                                           uint32_t keep_days, uint32_t max_size,
                                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        asyncCreateSharedFolderWithArchivation(fid, type, keep_days, max_size, init.handler);
        return init.result.get();
    }
protected:
    virtual void asyncCreateSharedFolder(Fid fid, OnUpdate hook) const = 0;
    virtual void asyncAddSubscriber(Fid fid, const std::string& subscriberUid, OnUpdate hook) const = 0;
    virtual void asyncGetAllSharedFolders(OnFidVecReceive hook) const = 0;
    virtual void asyncCreateSharedFolderWithArchivation(Fid fid,
                                                        const Folder::ArchivationType& type,
                                                        uint32_t keep_days, uint32_t max_size,
                                                        OnUpdate hook) const = 0;
};

typedef std::shared_ptr<SharedFoldersRepository> SharedFoldersRepositoryPtr;

} //namespace macs

#endif // MACS_SHARED_FOLDERS_REPOSITORY_H
