#ifndef MACS_SUBSCRIBED_FOLDERS_REPOSITORY_H
#define MACS_SUBSCRIBED_FOLDERS_REPOSITORY_H

#include <macs/types.h>
#include <macs/io.h>
#include <macs_pg/subscribed_folders/hooks.h>
#include <macs/threads_meta.h>

namespace macs {

using FidVec = std::vector<Fid>;
using OnFidVecReceive = Hook<FidVec>;

class SubscribedFoldersRepository : public std::enable_shared_from_this<SubscribedFoldersRepository> {
public:
    virtual ~SubscribedFoldersRepository() = default;
    template <typename Handler = io::sync_context>
    auto addFolder(Fid fid,
                   std::string ownerUid,
                   Fid ownerFid,
                   Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscribedFolder> init(handler);
        asyncAddFolder(fid, ownerUid, ownerFid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFolder(std::string ownerUid,
                   Fid ownerFid,
                   Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscribedFolder> init(handler);
        asyncGetFolder(ownerUid, ownerFid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeFolders(std::string ownerUid,
                       std::vector<Fid> ownerFids,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFidVecReceive> init(handler);
        asyncRemoveFolders(ownerUid, ownerFids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteMessages(std::string ownerUid,
                        Fid ownerFid,
                        MidVec ownerMids,
                        Revision ownerRevision,
                        Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateMessages> init(handler);
        asyncDeleteMessages(ownerUid, ownerFid, ownerMids, ownerRevision, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto labelMessages(const std::string& ownerUid,
                       const Fid& ownerFid,
                       const MidVec& ownerMids,
                       const Revision& ownerRevision,
                       const std::vector<Label>& labels,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateMessages> init(handler);
        asyncAddLabels(ownerUid, ownerFid, ownerMids, ownerRevision, labels, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto unlabelMessages(const std::string& ownerUid,
                         const Fid& ownerFid,
                         const MidVec& ownerMids,
                         const Revision& ownerRevision,
                         const std::vector<Label>& labels,
                         Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateMessages> init(handler);
        asyncRemoveLabels(ownerUid, ownerFid, ownerMids, ownerRevision, labels, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getSyncedRevision(const std::string& ownerId,
                           const Fid& ownerFid,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnOptRevisionReceive> init(handler);
        asyncGetSyncedRevision(ownerId, ownerFid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto syncMessage(const std::string& ownerUid,
                     const Envelope& entry,
                     const MimeParts& mime,
                     const std::vector<Hash>& referenceHashes,
                     const Hash& inReplyToHash,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSyncMessage> init(handler);
        asyncSyncMessage(ownerUid, entry, mime, referenceHashes, inReplyToHash, NotificationMode::on, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto syncMessageQuiet(const std::string& ownerUid,
                          const Envelope& entry,
                          const MimeParts& mime,
                          const std::vector<Hash>& referenceHashes,
                          const Hash& inReplyToHash,
                          Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSyncMessage> init(handler);
        asyncSyncMessage(ownerUid, entry, mime, referenceHashes, inReplyToHash, NotificationMode::off, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto joinThreads(const std::string& ownerUid,
                     const Fid& ownerFid,
                     const ThreadId& ownerTid,
                     const std::vector<ThreadId>& ownerJoinTids,
                     const Revision& ownerRevision,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateMessages> init(handler);
        asyncJoinThreads(ownerUid, ownerFid, ownerTid, ownerJoinTids, ownerRevision, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getEnvelopes(const std::string& ownerUid,
                      const Fid& ownerFid,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnEnvelopeReceive> init(handler);
        asyncGetEnvelopes(ownerUid, ownerFid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFoldersByOwner(const std::string& ownerUid,
                           Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscribedFolders> init(handler);
        asyncGetFoldersByOwner(ownerUid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getSyncedMids(const std::string& ownerUid,
                       const Fid& ownerFid,
                       std::size_t rowCount,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnMidsReceive> init(handler);
        asyncGetSyncedMids(ownerUid, ownerFid, rowCount, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getLastSyncedImapId(const std::string& ownerUid,
                            const Fid& ownerFid,
                            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<int64_t>> init(handler);
        asyncGetLastSyncedImapId(ownerUid, ownerFid, init.handler);
        return init.result.get();
    }

protected:
    enum class NotificationMode { on, off };

    virtual void asyncAddFolder(Fid fid,
                                std::string ownerUid,
                                Fid ownerFid,
                                OnSubscribedFolder hook) const = 0;

    virtual void asyncGetFolder(std::string ownerUid,
                                Fid ownerFid,
                                OnSubscribedFolder hook) const = 0;

    virtual void asyncRemoveFolders(std::string ownerUid,
                                    std::vector<Fid> ownerFids,
                                    OnFidVecReceive hook) const = 0;

    virtual void asyncDeleteMessages(std::string ownerUid,
                                     Fid ownerFid,
                                     MidVec ownerMids,
                                     Revision ownerRevision,
                                     OnUpdateMessages hook) const = 0;

    virtual void asyncAddLabels(const std::string& ownerUid,
                                const Fid& ownerFid,
                                const MidVec& ownerMids,
                                const Revision& ownerRevision,
                                const std::vector<Label>& labels,
                                OnUpdateMessages hook) const = 0;

    virtual void asyncRemoveLabels(const std::string& ownerUid,
                                   const Fid& ownerFid,
                                   const MidVec& ownerMids,
                                   const Revision& ownerRevision,
                                   const std::vector<Label>& labels,
                                   OnUpdateMessages hook) const = 0;

    virtual void asyncGetSyncedRevision(const std::string& ownerId,
                                        const Fid& ownerFid,
                                        OnOptRevisionReceive hook) const = 0;

    virtual void asyncSyncMessage(const std::string& ownerUid,
                                  const Envelope& entry,
                                  const MimeParts& mime,
                                  const std::vector<Hash>& referenceHashes,
                                  const Hash& inReplyToHash,
                                  NotificationMode notificationMode,
                                  OnSyncMessage hook) const = 0;

    virtual void asyncJoinThreads(const std::string& ownerUid,
                                  const Fid& ownerFid,
                                  const ThreadId& ownerTid,
                                  const std::vector<ThreadId>& ownerJoinTids,
                                  const Revision& ownerRevision,
                                  OnUpdateMessages hook) const = 0;

    virtual void asyncGetEnvelopes(const std::string& ownerUid,
                                   const Fid& ownerFid,
                                   OnEnvelopeReceive hook) const = 0;

    virtual void asyncGetFoldersByOwner(const std::string& ownerUid,
                                        OnSubscribedFolders hook) const = 0;

    virtual void asyncGetSyncedMids(const std::string& ownerUid,
                                    const Fid& ownerFid,
                                    std::size_t rowCount,
                                    OnMidsReceive hook) const = 0;

    virtual void asyncGetLastSyncedImapId(const std::string& ownerUid,
                                    const Fid& ownerFid,
                                    Hook<int64_t> hook) const = 0;
};

typedef std::shared_ptr<SubscribedFoldersRepository> SubscribedFoldersRepositoryPtr;

} //namespace macs

#endif // MACS_SUBSCRIBED_FOLDERS_REPOSITORY_H
