#ifndef MACS_SUBSCRIPTION_REPOSITORY_H_07032017
#define MACS_SUBSCRIPTION_REPOSITORY_H_07032017

#include <chrono>

#include <macs/io.h>
#include <macs_pg/subscription/hooks.h>
#include <macs_pg/subscription/subscription_action.h>

namespace macs {


class SubscriptionRepository : public std::enable_shared_from_this<SubscriptionRepository> {
public:
    using Action = macs::pg::SubscriptionAction;

    virtual ~SubscriptionRepository() = default;

    template <typename Handler = io::sync_context>
    auto findAJob(std::chrono::seconds aliveTimeout,
                  std::string launchId,
                  std::string hostname,
                  std::string workerVersion,
                  Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFoundJob> init(handler);
        asyncFindAJob(std::move(aliveTimeout), std::move(launchId), std::move(hostname), std::move(workerVersion), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto confirmTheJob(WorkerId workerId,
                       std::string launchId,
                       Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnJobConfirmation> init(handler);
        asyncConfirmTheJob(std::move(workerId), std::move(launchId), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getByWorker(WorkerId workerId,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionsChunkReceive> init(handler);
        asyncGetByWorker(std::move(workerId), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getById(std::string uid,
                 SubscriptionId subscriptionId,
                 Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionReceive> init(handler);
        asyncGetById(std::move(uid), subscriptionId, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto transitState(std::string uid,
                      SubscriptionId subscriptionId,
                      Action action,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionReceive> init(handler);
        asyncTransitState(std::move(uid), subscriptionId, action, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto transitState(SubscriptionId subscriptionId,
                      Action action,
                      Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionReceive> init(handler);
        asyncTransitState(subscriptionId, action, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto markFailed(std::string uid,
                    SubscriptionId subscriptionId,
                    std::string failReason,
                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionReceive> init(handler);
        asyncMarkFailed(std::move(uid), subscriptionId, std::move(failReason), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getFreeForWorker(WorkerId workerId,
                          std::size_t limit,
                          Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionsChunkReceive> init(handler);
        asyncGetFreeForWorker(std::move(workerId), limit, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto release(std::string uid,
                 SubscriptionId subscriptionId,
                 WorkerId workerId,
                 Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionReceive> init(handler);
        asyncRelease(std::move(uid), subscriptionId, std::move(workerId), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getByFids(std::string subscriberUid,
                   std::vector<Fid> fids,
                   Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionsChunkReceive> init(handler);
        asyncGetByFids(std::move(subscriberUid), std::move(fids), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeChunk(std::vector<SubscriptionId> subscriptionIds,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionsChunkReceive> init(handler);
        asyncRemoveChunk(std::move(subscriptionIds), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto addUnsubscribeTask(UnsubscribeTask task,
                            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUnsubscribeTaskReceive> init(handler);
        asyncAddUnsubscribeTask(std::move(task), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getUnsubscribeTasks(std::size_t limit,
                             std::chrono::seconds aliveTimeout,
                             Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUnsubscribeTaskVecReceive> init(handler);
        asyncGetUnsubscribeTasks(limit, std::move(aliveTimeout), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeSubscriptionsAndTask(UnsubscribeTask task,
                                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSubscriptionsChunkReceive> init(handler);
        asyncRemoveSunscriptionsAndTask(std::move(task), init.handler);
        return init.result.get();
    }

protected:
    virtual void asyncFindAJob(std::chrono::seconds aliveTimeout,
                               std::string launchId,
                               std::string hostname,
                               std::string workerVersion,
                               OnFoundJob hook) const = 0;

    virtual void asyncConfirmTheJob(WorkerId workerId,
                                    std::string launchId,
                                    OnJobConfirmation hook) const = 0;

    virtual void asyncGetByWorker(WorkerId workerId,
                                  OnSubscriptionsChunkReceive hook) const = 0;

    virtual void asyncGetById(std::string uid,
                              SubscriptionId subscriptionId,
                              OnSubscriptionReceive hook) const = 0;

    virtual void asyncTransitState(std::string uid,
                                   SubscriptionId subscriptionId,
                                   Action action,
                                   OnSubscriptionReceive hook) const = 0;

    virtual void asyncTransitState(SubscriptionId subscriptionId,
                                   Action action,
                                   OnSubscriptionReceive hook) const = 0;

    virtual void asyncMarkFailed(std::string uid,
                                 SubscriptionId subscriptionId,
                                 std::string failReason,
                                 OnSubscriptionReceive hook) const = 0;

    virtual void asyncGetFreeForWorker(WorkerId workerId,
                                       std::size_t limit,
                                       OnSubscriptionsChunkReceive hook) const = 0;

    virtual void asyncRelease(std::string uid,
                              SubscriptionId subscriptionId,
                              WorkerId workerId,
                              OnSubscriptionReceive hook) const = 0;

    virtual void asyncGetByFids(std::string subscriberUid,
                                std::vector<Fid> fids,
                                OnSubscriptionsChunkReceive hook) const = 0;

    virtual void asyncRemoveChunk(std::vector<SubscriptionId> subscriptionIds,
                                  OnSubscriptionsChunkReceive hook) const = 0;

    virtual void asyncAddUnsubscribeTask(UnsubscribeTask task,
                                         OnUnsubscribeTaskReceive hook) const = 0;

    virtual void asyncGetUnsubscribeTasks(std::size_t limit,
                                          std::chrono::seconds aliveTimeout,
                                          OnUnsubscribeTaskVecReceive hook) const = 0;

    virtual void asyncRemoveSunscriptionsAndTask(UnsubscribeTask task,
                                                 OnSubscriptionsChunkReceive hook) const = 0;
};


typedef std::shared_ptr<SubscriptionRepository> SubscriptionRepositoryPtr;

} // namespace macs

#endif // MACS_SUBSCRIPTION_REPOSITORY_H_07032017
