# coding: utf-8

from __future__ import (unicode_literals,
                        print_function)
from argparse import (ArgumentParser,
                      ArgumentDefaultsHelpFormatter)
import os

from jinja2 import Template
from psycopg2 import connect


def get_cpp_keywods():
    cpp_keywords = '''auto    double  int struct
break   else    long    switch
case    enum    register    typedef
char    extern  return  union
const   float   short   unsigned
continue    for signed  void
default goto    sizeof  volatile
do  if  static  while
asm dynamic_cast    namespace   reinterpret_cast
bool    explicit    new static_cast
catch   false   operator    template
class   friend  private this
const_cast  inline  public  throw
delete  mutable protected   true
try typeid  typename    using
using   virtual wchar_t
'''
    words = (w.strip() for w in cpp_keywords.split())
    return set(w for w in words if w)

CPP_KEYWORDS = get_cpp_keywods()


def get_enum_values_from_db(dsn, enum_name):
    with connect(dsn) as conn:
        cur = conn.cursor()
        cur.execute('SELECT enum_range(NULL::%s)::text[]' % enum_name)
        return cur.fetchone()[0]


def read_template():
    with open(os.path.join(
        os.path.dirname(__file__),
        'generate_enums.tmpl.h'
    )) as fd:
        return fd.read()


def make_camel_case(name):
    '''
    >>> make_camel_case('foo')
    'foo'
    >>> make_camel_case('foo-bar')
    'fooBar'
    '''
    words = name.split('-')
    return ''.join(
        [words[0].lower()] + [w.capitalize() for w in words[1:]]
    )


def make_c_name(db_name):
    '''
    >>> make_c_name('foo-bar')
    'fooBar'
    >>> make_c_name('auto')
    'auto_'
    '''
    name = make_camel_case(db_name)
    if name in CPP_KEYWORDS:
        return name + '_'
    return name


def make_enums(db_enums):
    '''
    >>> make_enums(['foo', 'foo-bar', 'register'])
    [('foo', 'foo'), ('fooBar', 'foo-bar'), ('register_', 'register')]
    '''
    enum_items = []
    for db_name in db_enums:
        c_name = make_c_name(db_name)
        enum_items.append(
            (c_name, db_name)
        )
    return enum_items


def render(**template_args):
    tmpl = Template(read_template())
    code = tmpl.render(**template_args)
    return (code + '\n').encode('utf-8')  # jinja2 strip last \n


def main():
    parser = ArgumentParser(
        formatter_class=ArgumentDefaultsHelpFormatter
    )
    parser.add_argument(
        'TYPE_NAME',
        help='type name in postgre eg: mail.change_type',
    )
    parser.add_argument(
        'CLASS_NAME',
        help='class name in generated file'
    )
    parser.add_argument(
        '--dsn',
        default='postgres://localhost/maildb',
        help='postgre connection string'
    )

    args = parser.parse_args()
    print(
        render(
            enum_items=make_enums(
                get_enum_values_from_db(
                    args.dsn, args.TYPE_NAME)),
            class_name=args.CLASS_NAME,
        )
    )

if __name__ == '__main__':
    main()
