#include <internal/folder/folders_converter.h>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptors.hpp>

namespace macs {
namespace pg {

namespace {

class SortedFolderVector {
    std::vector<Folder> v;
    static bool less (const Folder & f1, const Folder & f2) { return f1.fid() < f2.fid(); };
public:
    SortedFolderVector() {}

    SortedFolderVector(std::vector<Folder> folders) : v(std::move(folders)) {
        if (!std::is_sorted(v.begin(), v.end(), less)) {
            std::sort(v.begin(), v.end(), less);
        }
    }

    auto begin() -> decltype(v.begin()) { return v.begin();}
    auto begin() const -> decltype(v.begin()) { return v.begin();}
    auto end() -> decltype(v.end()) { return v.end();}
    auto end() const -> decltype(v.end()) { return v.end();}

    Folder & operator[] (std::size_t i) { return v[i]; }
    const Folder & operator[] (std::size_t i) const { return v[i]; }

    auto size() const -> decltype(v.size()) {return v.size();}

    auto find (const Fid& fid) const -> decltype(begin()) {
        const auto i = std::lower_bound(begin(), end(), FolderFactory().fid(fid).product(), less);
        return i == end() || fid != i->fid() ? end() : i;
    }
};

std::size_t parent(const SortedFolderVector& fs, std::size_t child) {
    return static_cast<size_t>(std::distance(fs.begin(), fs.find(fs[child].parentId())));
}

using ParentIndexVector = std::vector<std::size_t>;
using BadFolderVector = std::vector<Folder>;

ParentIndexVector getParents(const SortedFolderVector& folders) {
    const std::size_t nil = folders.size();
    ParentIndexVector parents(folders.size(), nil);
    for (std::size_t i = 0; i != folders.size(); ++i) {
        parents[i] = parent(folders, i);
    }
    return parents;
}

template<typename States>
std::tuple<SortedFolderVector, ParentIndexVector, BadFolderVector> removeBadFolders(
        SortedFolderVector folders, ParentIndexVector parents,
        const States& states, typename States::value_type badState) {
    if (std::find(states.begin(), states.end(), badState) == states.end()) {
        return std::make_tuple(std::move(folders), std::move(parents), BadFolderVector());
    } else {
        std::vector<Folder> goodFolders;
        BadFolderVector badFolders;
        for (std::size_t i = 0; i != states.size(); ++i) {
            if (states[i] == badState) {
                badFolders.emplace_back(std::move(folders[i]));
            } else {
                goodFolders.emplace_back(std::move(folders[i]));
            }
        };
        SortedFolderVector res(std::move(goodFolders));
        ParentIndexVector resParents = getParents(res);

        return std::make_tuple(std::move(res), std::move(resParents), std::move(badFolders));
    }
}

std::tuple<SortedFolderVector, ParentIndexVector, BadFolderVector> removeCycles(
        SortedFolderVector folders, ParentIndexVector parents) {
    enum State { unknown, ok, cycled };
    std::vector<State> states(folders.size(), unknown);
    const std::size_t nil = folders.size();

    auto visited = [&] (std::size_t i) { return states[i] != unknown; };

    for (std::size_t i = 0; i != folders.size(); ++i) {
        if (!visited(i)) {
            auto j = i;
            for (; j != nil && !visited(j); j = parents[j]) {
                states[j] = cycled;
            }

            if (j == nil || states[j] == ok) {
                for (auto k = i; k != j; k = parents[k]) {
                    states[k] = ok;
                }
            }
        }
    }

    return removeBadFolders(folders, parents, states, cycled);
}

std::tuple<SortedFolderVector, ParentIndexVector, BadFolderVector> removeBadParents(
        SortedFolderVector folders, ParentIndexVector parents) {
    enum State { unknown, ok, bad };
    std::vector<State> states(folders.size(), unknown);

    const std::size_t nil = folders.size();

    auto isRoot = [&] (std::size_t i) { return folders[i].parentId() == Folder::noParent; };
    auto visited = [&] (std::size_t i) { return states[i] != unknown; };


    for (std::size_t i = 0; i != folders.size(); ++i) {
        if (!visited(i)) {
            std::vector<std::size_t> path;
            for (auto j = i; j != nil && !visited(j); j = parents[j]) {
                path.push_back(j);
            }

            const auto last = path.back();
            const auto parent = parents[last];
            const State s = isRoot(last) ? ok : (parent != nil ? states[parent] : bad);
            for (auto i : path) {
                states[i] = s;
            }
        }
    }

    return removeBadFolders(folders, parents, states, bad);
}

FolderSet makeFolderSet(SortedFolderVector folders) {
    auto res = std::make_shared<FoldersMap>();
    for (auto &f : folders) {
        const auto fid = f.fid();
        res->insert({fid, std::move(f)});
    }
    return FolderSet(std::move(res));
}

void logBadFolders(const BadFolderVector& folders, logging::v2::LogPtr log,
        const std::string& details, const std::string& reason) {
    if (log && !folders.empty()) {
        using namespace logging;
        using boost::algorithm::join;
        using boost::adaptors::transformed;
        const auto fid = [] (const Folder & f) {return f.fid(); };

        logError(*log, [&] {
            std::ostringstream s;
            s << "skip folders with fids=" << join(folders | transformed(fid), ",") << " due to " << reason;
            return Record{ details, s.str(), Attributes() };
        });
    }
}

}

FolderSet normalize(std::vector<Folder> fs, logging::v2::LogPtr log, const std::string& details) {
    SortedFolderVector allFolders(fs);
    ParentIndexVector allParents = getParents(allFolders);

    SortedFolderVector noCycleFolders;
    ParentIndexVector noCycleParents;
    BadFolderVector cycles;
    std::tie(noCycleFolders, noCycleParents, cycles) =
            removeCycles(std::move(allFolders), std::move(allParents));

    SortedFolderVector validFolders;
    ParentIndexVector validParents;
    BadFolderVector unknownParents;
    std::tie(validFolders, validParents, unknownParents) =
            removeBadParents(std::move(noCycleFolders), std::move(noCycleParents));

    logBadFolders(cycles, log, details, "cycles");
    logBadFolders(unknownParents, log, details, "unknown parents");

    return makeFolderSet(std::move(validFolders));
}

} // namespace pg
} // namespace macs
