#include <sharpei_client/integration.h>
#include <macs_pg/integration/common_configure.h>
#include <macs_pg/macs_pg.h>

namespace macs {
namespace pg {
namespace integration {

macs::pg::Credentials CommonConfigure::parseCredentials(const ptree& pgTree) {
    macs::pg::Credentials c;
    c.user = pgTree.get("user", "");
    c.password = pgTree.get("password", "");
    return c;
}

macs::pg::UidResolverFactoryPtr CommonConfigure::sharpei(sharpei::client::http::HttpClientPtr h,
        const sharpei::client::Settings& s,
        const macs::pg::Credentials& c) {
    if (!h && !s.httpClient) {
        throw std::invalid_argument("Need sharpei client");
    }
    const pgg::SharpeiParams params = {h, s, c};
    return pgg::createSharpeiUidResolverFactory(params);
}

macs::pg::UidResolverFactoryPtr CommonConfigure::directPg( const std::string& connString,
                                                           const std::string& shardName ) {
    return pgg::createFakeUidResolverFactory(connString, shardName);
}

macs::pg::UidResolverFactoryPtr CommonConfigure::parseUidResolverFactory(const ptree& tree,
        sharpei::client::http::HttpClientPtr h,
        sharpei::client::Settings::HttpExceptionHandler onHttpExc,
        sharpei::client::Settings::HttpErrorHandler onHttpErr) {
    try {
        const auto& pgTree = tree.get_child_optional("pg");
        if (pgTree) {
            const auto credentials = parseCredentials(*pgTree);
            const auto& sharpeiTree = tree.get_child_optional("sharpei");
            if (sharpeiTree) {
                const auto settings = sharpei::client::integration::parseSharpeiSettings(
                    *sharpeiTree, onHttpExc, onHttpErr);
                return sharpei(h, settings, credentials);
            } else {
                return directPg(pgTree->get<std::string>("connection_string"),
                                pgTree->get<std::string>("shard_name"));
            }
        } else {
            return macs::pg::UidResolverFactoryPtr();
        }
    } catch (const std::exception& e) {
        throw std::runtime_error("failed to init uid resolver factory settings: " + std::string(e.what()));
    }
}

macs::pg::ConnectionPoolPtr CommonConfigure::parseConnectionPool(const ptree& pgTree, boost::asio::io_context* ctx) {
    try {
        const auto poolTree = pgTree.get_child("connection_pool");
        using Ms = macs::pg::Milliseconds;
        using Sec = macs::pg::Seconds;
        auto factory = pgg::ConnectionPoolFactory()
            .connectTimeout(Ms(poolTree.get<unsigned long>("connect_timeout_ms")))
            .queueTimeout(Ms(poolTree.get<unsigned long>("queue_timeout_ms")))
            .queryTimeout(Ms(poolTree.get<unsigned long>("query_timeout_ms")))
            .maxConnections(poolTree.get<std::size_t>("max_connections"));
        if (ctx) {
            factory.ioService(*ctx);
        } else {
            factory.workersCount(poolTree.get<std::size_t>("workers_count"));
        }
        if (auto asyncResolve = poolTree.get_optional<bool>("async_resolve")) {
            factory.asyncResolve(*asyncResolve);
        }
        if (auto ipv6Only = poolTree.get_optional<bool>("ipv6_only")) {
            factory.ipv6Only(*ipv6Only);
        }
        if (auto dnsCacheTTL = pgTree.get_optional<unsigned long>("dns.cache_ttl")) {
            factory.dnsCacheTTL(Sec(*dnsCacheTTL));
        }
        return factory.product();
    } catch (const std::exception& e) {
        throw std::runtime_error("failed to init pg connection pool: " + std::string(e.what()));
    }
}

macs::pg::QueryConf CommonConfigure::parseQueryConf(const ptree& pgTree) {
    try {
        return macs::pg::readQueryConfFile( pgTree.get<std::string>("query_conf") );
    } catch (const std::exception& e) {
        throw std::runtime_error("failed to init pg query config: " + std::string(e.what()));
    }
}

CommonConfigure& CommonConfigure::parse(const ptree& tree, boost::asio::io_context* ctx) {
    try {
        const auto& pgTree = tree.get_child_optional("pg");
        if (pgTree) {
            credentials(parseCredentials(*pgTree));
            queryConf(parseQueryConf(*pgTree));
            transactionTimeout_ = Milliseconds(pgTree->get<unsigned long>("transaction_timeout_ms", 0));
            connPool(parseConnectionPool(*pgTree, ctx));
        }
    } catch (const std::exception& e) {
        throw std::runtime_error("failed to init pg settings: " + std::string(e.what()));
    }
    return *this;
}

CommonConfigure& CommonConfigure::parse(const boost::property_tree::ptree& tree) {
    return parse(tree, nullptr);
}

CommonConfigure& CommonConfigure::queryConf( macs::pg::QueryConf v ) {
    conf = v;
    return *this;
}

CommonConfigure& CommonConfigure::connPool( macs::pg::ConnectionPoolPtr v ) {
    pool = v;
    return *this;
}

CommonConfigure& CommonConfigure::credentials( macs::pg::Credentials v ) {
    credentials_ = std::move(v);
    return *this;
}

macs::pg::ServiceFactoryPtr CommonConfigure::getFactory(UidResolverFactoryPtr resolverFactory,
        ShardResolverFactoryPtr shardResolverFactory) const {
    if( conf == nullptr ) {
        throw std::logic_error( "CommonConfigure::getFactory: QueryConf object wasn't set" );
    }
    if( pool == nullptr ) {
        throw std::logic_error( "CommonConfigure::getFactory: ConnectionPool object wasn't set" );
    }
    macs::pg::ServiceFactoryPtr pgFactory = macs::pg::createSeviceFactory(pool, resolverFactory, shardResolverFactory);
    pgFactory->queryConf( conf ).credentials( credentials_ ).transactionTimeout(transactionTimeout_);
    return pgFactory;
}

}}}
