#include <internal/tab/converter.h>
#include <internal/reflection/tab.h>

namespace macs {
namespace pg {

namespace {

using Tabs = std::vector<Tab>;
using UnknownTabTypes = std::vector<std::string>;

std::tuple<Tabs, UnknownTabTypes> fromReflection(std::vector<reflection::Tab> data) {
    Tabs goodTabs;
    UnknownTabTypes badTypes;

    for (const auto& tref: data) {
        Tab t = TabFactory().fromReflection(tref).release();
        if (t.type() == Tab::Type::unknown) {
            badTypes.emplace_back(tref.tab);
        } else {
            goodTabs.emplace_back(std::move(t));
        }
    }

    return std::make_tuple(std::move(goodTabs), std::move(badTypes));
}

void logUnknownTabs(UnknownTabTypes types, logging::v2::LogPtr log,
                    const std::string& details) {
    if (log && !types.empty()) {
        using namespace logging;
        using boost::algorithm::join;

        logError(*log, [&] {
            std::ostringstream s;
            s << "skip tabs with unknown types=" << join(types, ",");
            return Record{ details, s.str(), Attributes() };
        });
    }
}

TabSet makeTabSet(Tabs tabs) {
    auto res = std::make_shared<TabsMap>();
    for (auto &t : tabs) {
        const auto type = t.type();
        res->insert({type, std::move(t)});
    }
    return TabSet(std::move(res));
}

}

TabSet makeTabSet(std::vector<reflection::Tab> data,
                  logging::v2::LogPtr log,
                  const std::string& details) {
    auto [tabs, unknownTabs] = fromReflection(std::move(data));

    logUnknownTabs(std::move(unknownTabs), log, details);

    return makeTabSet(std::move(tabs));
}

} // namespace macs
} // namespace pg
