#include <future>

#include "service/get_macs_service.h"
#include "service/options.h"

#include <macs/folder.h>

using namespace macs::pg::utils;

using Symbol = macs::Folder::Symbol;

int main(int argc, char **argv) {
    std::string uid, dsn, name, parent, symbolTitle;

    const auto err = parse(argc, argv).as(Uid(uid), Dsn(dsn),
                                          FolderName(name),
                                          ParentFid(parent).def(macs::Folder::noParent),
                                          FolderSymbol(symbolTitle).def(Symbol::none.title()));

    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }

    const auto service = getMacsService(uid, dsn);

    std::cout << "# Create folder with name=\"" << name << "\"";

    if (!parent.empty()) {
        std::cout << " in folder with fid=" << parent;
    }

    if (!symbolTitle.empty()) {
        std::cout << " with symbol=" << symbolTitle;
    }

    const auto& symbol = Symbol::getByTitle(symbolTitle);
    if (!symbolTitle.empty() && symbol == Symbol::none) {
        std::cerr << "No such symbol:" << symbolTitle << std::endl;
        return -1;
    }

    std::cout << " for user with uid=" << uid << std::endl;

    std::promise<void> result;
    auto f = result.get_future();

    const auto handle = [&result] (const macs::error_code& ec,
            const macs::Folder& folder) {
        auto p = std::move(result);
        if (ec) {
            mail_errors::setRuntimeError(p, ec);
        } else {
            std::cout << "- Revision: " << folder.revision() << std::endl
                      << "  Fid: " << folder.fid() << std::endl;
            p.set_value();
        }
    };

    try {
        service->folders().createFolder(name, parent, symbol, handle);
        f.get();
    } catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }

    return 0;
}
