#ifndef MACS_PG_UTILS_ENVELOPE_IO_H_
#define MACS_PG_UTILS_ENVELOPE_IO_H_

#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <macs_pg/macs_pg.h>
#include "envelope_print_scope.h"
#include <iostream>

namespace macs {
namespace pg {
namespace utils {

class EnvelopePrinter {
public:
    EnvelopePrinter(std::ostream & s, EnvelopePrintScope scope = envelopePrintAllMailInfo)
    : s(s), printScope(scope) {}

    void print( const macs::Envelope & e) const {
        const std::string * indent = startIndent();
        static const std::string fakeAttachLid = "FAKE_ATTACHED_LBL";
        static const std::string fakeSeenLid = "FAKE_SEEN_LBL";

        printField(envelopePrintMid, "Mid", e.mid(), indent);
        printField(envelopePrintFid, "Fid", e.fid(), indent);
        printField(envelopePrintTid, "Tid", e.threadId(), indent);
        printField(envelopePrintRevision, "Revision", e.revision(), indent);
        printField(envelopePrintSentDate, "SentDate", e.date(), indent);
        printField(envelopePrintReceiveDate, "ReceiveDate", e.receiveDate(), indent);
        printField(envelopePrintFrom, "From", e.from(), indent);
        printField(envelopePrintTo, "To", e.to(), indent);
        printField(envelopePrintCc, "Cc", e.cc(), indent);
        printField(envelopePrintReplyTo, "ReplyTo", e.replyTo(), indent);
        printField(envelopePrintSubject, "Subject", e.subject(), indent);
        printField(envelopePrintTypes, "Types", e.types(), indent);
        printField(envelopePrintLabels, "Labels", e.labels(), indent);
        printField(envelopePrintSeen, "Seen", e.hasLabel(fakeSeenLid), indent);
        printField(envelopePrintHasAttaches, "HasAttachments", e.hasLabel(fakeAttachLid), indent);
        printField(envelopePrintThreadCount, "ThreadCount", e.threadCount(), indent);
        printField(envelopePrintThreadNewCount, "ThreadNewCount", e.newCount(), indent);
    }

private:
    std::ostream & s;
    EnvelopePrintScope printScope;

    static const std::string * startIndent() {
        static const std::string val = "- ";
        return &val;
    }

    static const std::string * commonIndent() {
        static const std::string val = "  ";
        return &val;
    }

    template <typename ValueT>
    void printField(const EnvelopePrintScope & scope, const std::string & name,
            const ValueT & value, const std::string * & indent) const {
        if (printScope ^ scope) {
            s << *indent << name << ": ";
            stream(value);
            s << std::endl;
            indent = commonIndent();
        }
    }

    template <typename ValueT>
    void stream(const ValueT & value) const {
        s << value;
    }
};

std::string escapeYAMLQuotedString(const std::string &value) {
    std::string es;
    for (const auto chr : value) {
        switch (chr) {
            case '\n':
                es += "\\n";
            case '\r':
                es += "\\r";
            case '\\':
                es += "\\\\";
            case '\"':
                es += "\\\"";
            default:
                es += chr;
            }
    }
    return es;
}

std::string escapeYAMLString(const std::string & value) {
    // string starts_with ' " or contains \r\n
    if (value.find_first_of("\"'") == 0 || value.find_first_of("\n\r") != std::string::npos) {
        return std::string("\"") + escapeYAMLQuotedString(value) + std::string("\"");
    } else {
        return value;
    }
}

template <>
inline void EnvelopePrinter::stream(const std::string & value) const {
    s << escapeYAMLString(value);
}

template <>
inline void EnvelopePrinter::stream(const bool & value) const {
    s << (value ? "yes" : "no");
}

template <>
inline void EnvelopePrinter::stream(const std::vector< std::string > & value) const {
    s << "[" << boost::join(value | boost::adaptors::transformed(escapeYAMLString) , "," ) << "]";
}

template <>
inline void EnvelopePrinter::stream(const std::set<int>& value) const {
    s << "[" << boost::algorithm::join( value | boost::adaptors::transformed(
            boost::bind(boost::lexical_cast<std::string,int>,_1)), ",") << "]";
}

} // namespace utils
} // namespace pg
} // namespace macs



#endif /* MACS_PG_UTILS_ENVELOPE_IO_H_ */
