#include <macs_pg/macs_pg.h>
#include <boost/format.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/assign.hpp>
#include "service/get_macs_service.h"
#include "service/options.h"
#include "service/formatter.h"

#include <set>

struct FoldersCompare{
    bool operator()(const macs::Folder &a, const macs::Folder &b)
    {
        if (a.symbolicName() != macs::Folder::Symbol::none || b.symbolicName() != macs::Folder::Symbol::none) {
            return boost::lexical_cast<int>(a.fid()) <= boost::lexical_cast<int>(b.fid());
        }
        return a.name() <= b.name();
    }
};

using namespace macs::pg::utils;

int main(int argc, char **argv) {
    std::string uid, dsn;

    const auto err = parse(argc, argv).as(Uid(uid), Dsn(dsn));
    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }


    macs::ServicePtr service( getMacsService(uid, dsn));

    typedef std::set<macs::Folder, FoldersCompare> FoldersList;

    const macs::FolderSet folderSet = service->folders().getAllFolders();

    FoldersList folders;
    boost::copy(
        folderSet | boost::adaptors::map_values,
        std::inserter(folders, folders.begin())
    );
    const std::string fmtString = makeFormatStr({
        "Fid", "ParentId", "SymbolicName",
        "MessagesCount", "NewMessagesCount", "RecentMessagesCount", "Unvisited",
        "Revision", "Name"});
    for (FoldersList::const_iterator fldr = folders.begin(); fldr!=folders.end(); ++fldr) {
        std::cout << boost::format(fmtString)
            % fldr->fid()
            % fldr->parentId()
            % fldr->symbolicName().title()
            % fldr->messagesCount()
            % fldr->newMessagesCount()
            % fldr->recentMessagesCount()
            % fldr->unvisited()
            % fldr->revision()
            % fldr->name();
    }
    return 0;
}
