#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <macs_pg/macs_pg.h>

#include "envelope_io.h"
#include "service/get_macs_service.h"
#include "service/options.h"

using namespace macs::pg::utils;

int main(int argc, char **argv) {
    std::string uid, dsn, fid;
    int maxImapId, chunkSize;

    const auto err = parse(argc, argv).as(Uid(uid), Dsn(dsn), Fid(fid),
                                    MaxImapId(maxImapId).def(1000000),
                                    ChunkSize(chunkSize).def(1000000));

    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }

    std::cout << "# **** Execute custom MailboxListFolders request for UID=" << uid
              << " and FID=" << fid << std::endl;

    macs::ServicePtr service = getMacsService(uid, dsn);
    auto reader = service->envelopes().query().withMimes().inFolder(fid)
            .byChunks(chunkSize).maxImapId(maxImapId)
            .chunkReader();

    macs::EnvelopeWithMimeChunk chunk;
    std::size_t chunkCount = 0;
    std::size_t totalFetched = 0;
    do {
        macs::error_code ec;
        chunk = reader.next(macs::io::use_sync[ec]);
        if (ec) {
            std::cerr << "Got error: " << ec.message() << std::endl;
            break;
        } else {
            if (!chunk.empty()) {
                ++chunkCount;
                totalFetched += chunk.size();

                std::cout << "Chunk#" << chunkCount
                          << " of size " << chunk.size()
                          << std::endl;
                for (const auto& v: chunk) {
                    const macs::Envelope& e = std::get<macs::Envelope>(v);
                    EnvelopePrinter(std::cout, envelopePrintAllMailInfo).print(e);
                }
            }
        }
    } while (!chunk.empty());

    std::cout << "Total fetched: " << totalFetched
              << " with " << chunkCount << " chunks" << std::endl;

    return 0;
}
