#include <macs_pg/macs_pg.h>

#include "service/get_macs_service.h"
#include "service/options.h"

using namespace macs::pg::utils;

inline macs::ImapFolder convertFolder(const macs::Folder& folder)
{
    macs::ImapFolder result;
    result.revision = folder.revision();
    result.messageCount = folder.messagesCount();
    result.recentCount = folder.recentMessagesCount();
    result.unseenCount = folder.newMessagesCount();
    result.uidNext = folder.imapUidNext();
    result.uidValidity = folder.imapUidValidity();
    result.name = folder.name();
    result.fid = folder.fid();
    return result;
}

namespace {
std::ostream & operator << ( std::ostream & s,  const macs::ImapUidMapRecord & v ) {
    s << "- Uid: "<< v.uid << "\n  Chain: " << v.chain;
    return s;
}

std::ostream & operator << ( std::ostream & s,  const macs::ImapEnvelope& v ) {
    s << "- Uid: "<< v.uid << "\n  Num: " << v.num << "\n  Mid: " << v.mid << "\n  Revision: " << v.modseq;
    return s;
}

std::ostream & operator << ( std::ostream & s,  const macs::ImapEnvelopeChanges& v ) {
    s << v.envelope << "\n  baseId: " << v.baseId << "\n  Offset: " << v.offset << "\n  Deleted: " << v.deleted;
    return s;
}

std::ostream & operator << ( std::ostream & s,  const macs::ImapEnvelopeDetails& v ) {
    s << v.envelope << "\n  StId: " << v.stid << "\n  Mid: " << v.mid << "\n  Size: " << v.size;
    return s;
}
}

template <typename T>
struct OnIMAP {
    size_t errorsCount;

    OnIMAP(const std::string queryDescription) {
        std::cout << "# ===============================================" << std::endl;
        std::cout << "# Begin request " << queryDescription << std::endl;
        std::cout << "# ===============================================" << std::endl;
    }

    void operator()(macs::error_code const & err, T const &imapData) const {
        if( err ) {
            std::cerr << "code: " << err.value() << " message: " <<  err.message() << std::endl;
        } else {
              for (auto it = imapData.begin(); it != imapData.end(); ++it) {
                 std::cout << *(it) << std::endl;
                 std::cout << "# -----------------------------------------------" << std::endl;
             }
        }
    }
};


int main(int argc, char **argv) {
    std::string uid, dsn, fid;
    bool deleted;
    size_t changes = 0;
    uint64_t fromNum = 0;
    int fromCount = 10;
    std::vector<uint64_t> details;

    const auto flagOpt = Flag(deleted, "deleted", "only deleted messages");
    const auto fromOpt = Uint64T(fromNum, "from_num", "get --from-count messages starting --from-num").notNecessarily();
    const auto countOpt = Count(fromCount).notNecessarily();
    const auto changesOpt = SizeT(changes, "changes", "changes from given revision").notNecessarily();
    const auto detailsOpt = VecUint64(details, "details,d", "get details by messages UIDs").notNecessarily();

    const auto res = parse(argc, argv).as(Uid(uid), Dsn(dsn), Fid(fid),
                                          flagOpt, changesOpt, fromOpt, countOpt, detailsOpt);
    if (res) {
        std::cerr << res.message() << std::endl;
        return -1;
    }

    macs::ServicePtr service = getMacsService(uid, dsn);
    macs::ImapFolder folder = convertFolder(service->folders().getFolderByFid(fid));

    std::cout <<
        "# ****** Execute IMAP queries for UID=" << uid <<
        " and FID=" << fid << std::endl;

    macs::error_code ec;

    if (changes > 0) {
        folder.revision = changes;
        auto data = service->imapRepo().imapGetChanges(
             folder,
             macs::io::use_sync[ec]
        );
        OnIMAP<macs::ImapEnvelopeChangesChunk>("Changes")(ec, data);
    } else if (deleted) {
        auto data = service->imapRepo().imapGetDeleted(
            folder,
            macs::io::use_sync[ec]
        );
        OnIMAP<macs::ImapEnvelopeChunk>("Deleted")(ec, data);
    } else if (fromNum > 0) {
        const uint64_t low = fromNum;
        const uint64_t hi = low + fromCount;

        auto data = service->imapRepo().imapGetMails(
            folder, low, hi,
            macs::ImapRepository::ImapListMode_byNum,
            macs::io::use_sync[ec]
        );
        OnIMAP<macs::ImapEnvelopeChunk>("Mails")(ec, data);
    } else if (!details.empty()) {
        auto data = service->imapRepo().imapGetDetails(
            folder, details,
            macs::io::use_sync[ec]
        );
        OnIMAP<macs::ImapEnvelopeDetailsChunk>("Details")(ec, data);
    } else {
        auto data = service->imapRepo().imapGetUidMap(
            folder,
            macs::io::use_sync[ec]
        );
        OnIMAP<macs::ImapUidMapChunk>("UidMap")(ec, data);
    }
    return 0;
}
