#pragma once

#include "handlers_helper.h"
#include "service/get_macs_service.h"
#include "service/options.h"
#include <boost/algorithm/string/join.hpp>

namespace mimes {

using namespace macs::pg::utils;

void printMimePart(const macs::MimePart& mimePart) {
    std::cout << "  - hid: " << mimePart.hid()
            << "\n    contentType: " << mimePart.contentType()
            << "\n    contentSubtype: " << mimePart.contentSubtype()
            << "\n    boundary: " << mimePart.boundary()
            << "\n    name: " << mimePart.name()
            << "\n    charset: " << mimePart.charset()
            << "\n    encoding: " << mimePart.encoding()
            << "\n    contentDisposition: " << mimePart.contentDisposition()
            << "\n    fileName: " << mimePart.fileName()
            << "\n    cid: " << mimePart.cid()
            << "\n    offsetBegin: " << mimePart.offsetBegin()
            << "\n    offsetEnd: " << mimePart.offsetEnd() << std::endl;
}

template <class GetMimes>
int main(int argc, char** argv, const char* target, GetMimes getMimes) {
    std::string uid, dsn;
    std::vector<std::string> mids;
    const auto err = parse(argc, argv).as(Uid(uid), Dsn(dsn), Mids(mids));
    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }

    const auto service = getMacsService(uid, dsn);
    std::cout << "# Get " << target << " mimes for uid=" << uid
            << " and MIDS={" << boost::algorithm::join(mids, ",") << "}" << std::endl;

    try {
        macs::Mids midsList;
        std::copy(mids.begin(), mids.end(), std::back_inserter(midsList));
        const auto midsWithMimes = getMimes(service->envelopes(), midsList);
        for (const auto& midWithMime: midsWithMimes) {
            std::cout << "- mid: " << std::get<0>(midWithMime)
                    << "\n  stid: " << std::get<1>(midWithMime)
                    << "\n  mime:" << std::endl;
            for (const auto& mimePart: std::get<2>(midWithMime)) {
                printMimePart(mimePart);
            }
        }
    } catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
        return -1;
    }

    return 0;
}

} // namespace
