#include <future>

#include <boost/algorithm/string/join.hpp>
#include <yamail/data/deserialization/json_reader.h>
#include <internal/envelope/mime_parts.h>

#include <service/get_macs_service.h>
#include <service/options.h>

#include <macs/envelope.h>
#include "envelope_io.h"
#include <mail_errors/set_exception.h>

using namespace macs::pg::utils;

inline std::ostream& operator<<(std::ostream& os, macs::EnvelopeKind kind) {
    switch (kind) {
    case macs::EnvelopeKind::original:
        os << "original";
        break;
    case macs::EnvelopeKind::duplicate:
        os << "duplicate";
        break;
    }

    return os;
}

int main(int argc, char **argv) {
    using boost::algorithm::join;

    std::string uid, dsn, tid, fid, subject, pop_uidl, st_id, tabType,
                firstline, hdr_message_id, extra_data, from, to, mimeJson;
    std::size_t size;
    std::time_t hdr_date, received_date;
    bool storeDeleted = false;

    const auto err = parse(argc, argv).as(Uid(uid), Dsn(dsn),
                                          Tid(tid),
                                          Fid(fid),
                                          Time(hdr_date, "hdr_date", "sent date in unix time"),
                                          Time(received_date, "received_date", "received date in unix time"),
                                          String(subject, "subject", "subject string"),
                                          String(pop_uidl, "pop_uidl", "uidl"),
                                          Stid(st_id),
                                          String(firstline, "firstline", "firstline string"),
                                          String(hdr_message_id, "hdr_message_id", "firstline string"),
                                          SizeT(size, "size", "message size in bytes"),
                                          String(extra_data, "extra_data", "extra data string"),
                                          String(mimeJson, "mime", "mime parts json"),
                                          String(from, "from", "from address"),
                                          String(to, "to", "to address"),
                                          StoreDeleted(storeDeleted),
                                          Tab(tabType));
    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }

    std::optional<macs::Tab::Type> tab;
    if (!tabType.empty()) {
        tab = std::make_optional(macs::Tab::Type::fromString(tabType));
    }

    const auto service = getMacsService(uid, dsn);
    auto f = macs::EnvelopeFactory().threadId(tid)
            .fid(fid).date(hdr_date).receiveDate(received_date)
            .subject(subject).uidl(pop_uidl).stid(st_id)
            .firstline(firstline).rfcId(hdr_message_id).size(size)
            .extraData(extra_data).from(from).to(to).tab(tab).release();

    std::cout << "#Create envelope for the user with uid=" << uid << std::endl;

    macs::EnvelopesRepository::SaveOptions saveOptions;
    saveOptions.ignoreDuplicates = false;
    if (storeDeleted) {
        saveOptions.notificationMode = macs::EnvelopesRepository::NotificationMode::off;
        saveOptions.storeType = macs::EnvelopesRepository::StoreType::deleted;
    } else {
        saveOptions.notificationMode = macs::EnvelopesRepository::NotificationMode::on;
        saveOptions.storeType = macs::EnvelopesRepository::StoreType::box;
    }

    try {
        auto mimeR = yamail::data::deserialization::fromJson<
                std::vector<macs::pg::reflection::MimePart>>(mimeJson);
        auto mime = macs::pg::toMime(std::move(mimeR));
        macs::error_code err;
        auto uer = service->envelopes().save(f, std::move(mime),
                [] (const macs::Envelope&) { return macs::ThreadMeta {}; },
                saveOptions,
                macs::io::use_sync[err]);

        if (err) {
            throw std::runtime_error(std::string(err.category().name()) + ": " + err.message());
        } else {
            const auto& e = std::get<macs::Envelope>(uer);
            EnvelopePrinter(std::cout).print(e);
            std::cout << "  Revision: " << e.revision() << std::endl;
            std::cout << "  Kind: " << std::get<macs::EnvelopeKind>(uer) << std::endl;
        }
    } catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }

    return 0;
}


