#ifndef MACS_PG_UTILS_SERVICE_OPTIONS_H
#define MACS_PG_UTILS_SERVICE_OPTIONS_H

#include <boost/range/algorithm/copy.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/program_options.hpp>
#include <string>
#include <vector>
#include <list>
#include <iostream>

#include <service/params.h>
#include <macs/types.h>

namespace macs {
namespace pg {
namespace utils {

namespace po = boost::program_options;
struct Data {
    po::options_description options_;
    po::variables_map vm_;
    std::vector<std::string> required_;

    Data() : options_("Allowed options") { }
};

template<int level, typename Tuple, typename... Tail> struct Holder;

template<int level, typename Tuple>
struct Holder<level, Tuple> {
    typedef Tuple Args;
    static void addOption(Data&, const Args&) { }
    static void parse(Data&, const Args&) { }
};

template<typename T>
void evalOptionValue(const std::string & name, const po::variables_map& vm_, SingleOption<T>& option) {
    auto val = vm_.count(name) ? vm_[name]. template as<T>() : option.defaultValue();
    option.set(val);
}

void evalOptionValue(const std::string & name, const po::variables_map& vm_, BoolOption& option) {
    if (vm_.count(name)) {
        option.set(true);
    }
}

template<int level, typename Tuple, typename Head, typename... Tail>
struct Holder<level, Tuple, Head, Tail...> : Holder<level+1, Tuple, Tail...> {
    typedef Holder<level+1, Tuple, Tail...> Next;
    typedef Tuple Args;
    typedef typename Head::Type Type;

    enum {
        Level = level
    };

    static std::string addLabel(const Head& opt) {
        return opt.required() ? (opt.description()+" (required)") : opt.description();
    }

    static void required(Data& d, const Head& opt) {
        if (opt.required()) {
            d.required_.push_back(opt.name());
        }
    }

    static void addOption(Data& d, const Args& args) {
        const auto& option = std::get<level>(args);
        const auto& name = option.fullName();
        const auto& value = po::value<Type>()->multitoken();
        const auto& descr = addLabel(option);

        if(option.isFlag()) {
            d.options_.add_options()(name.c_str(), descr.c_str());
        } else {
            d.options_.add_options()(name.c_str(), value, descr.c_str());
        }
        required(d, option);

        Next::addOption(d, args);
    }


    static void parse(Data& d, Args& args) {
        auto& option = std::get<level>(args);
        const auto& name = option.name();

        evalOptionValue(name, d.vm_, option);

        Next::parse(d, args);
    }
};

template<class... Args>
class Options : Holder<0, std::tuple<Args...>, Args...> {
    typedef std::tuple<Args...> Tuple;
    typedef Holder<0, Tuple, Args...> Base;

    int argc_;
    char** argv_;
    Data data_;

    void addHelp() {
        data_.options_.add_options()
                ("help,h", "produce help message");
    }

    Options(int argc, char** argv) : argc_(argc), argv_(argv) {
        addHelp();
    }

    void addOptions(Tuple& options) {
        Base::addOption(data_, options);
    }

    void parse(Tuple& options) {
        po::store(po::parse_command_line(argc_, argv_, data_.options_), data_.vm_);
        po::notify(data_.vm_);

        Base::parse(data_, options);
    }

    std::vector<std::string> satisfied() const {
        std::vector<std::string> missing;
        for(const auto& param : data_.required_) {
            if(!data_.vm_.count(param)) {
                missing.push_back(param);
            }
        }
        return missing;
    }

    friend class OptionMaker;

public:
    const po::options_description& description() const {
        return data_.options_;
    }

    std::set<std::string> parsedOptions() const {
        std::set<std::string> ret;
        boost::copy(data_.vm_ | boost::adaptors::map_keys, std::inserter(ret, ret.begin()));
        return ret;
    }

    bool needHelp() const {
        return argc_ <= 1 || data_.vm_.count("help");
    }
};

struct OptionCategory : public boost::system::error_category {
    enum Code {
        MissingRequiredOption = 1,
        UnknownOption,
        UnknownError
    };

    const char* name() const noexcept override {
        return "options parser";
    }

    std::string message(int value) const override {
        if (value == MissingRequiredOption) {
            return "Missing required option";
        } else if (value == MissingRequiredOption) {
            return "Unknown option";
        }

        return "options parser error";
    }
};

inline OptionCategory& getCategory() {
    static OptionCategory category;
    return category;
}

struct error_code_options : public error_code {
    std::set<std::string> opts;
    using base = error_code;

    error_code_options() {}
    error_code_options(int val, const error_category& cat) : base(val, cat) { }
    error_code_options(const std::set<std::string>& s) : opts(s) { }
    error_code_options(const base& b, const std::string& m) : base(b.base(), m) {}

    template<class Opt>
    bool has(const Opt& opt) const { return opts.find(opt.name()) != opts.end(); }
    bool has(const std::string& opt) const { return opts.find(opt) != opts.end(); }
};

class OptionMaker {
    int argc_;
    char** argv_;

    static error_code_options makeErrorCode(OptionCategory::Code err,
                                         const std::string& message)  {
        using namespace boost::system;
        return error_code_options(error_code(err, getCategory()), message);
    }

    template<class Opt>
    static std::string addDescription(const Opt& o, const std::string& message) {
        std::ostringstream out;
        out << message << "\n\n" << o.description() << std::endl;
        return out.str();
    }

public:
    OptionMaker(int argc, char** argv)
        : argc_(argc), argv_(argv)
    { }

    template<class ... Args>
    error_code_options as(const Args&... args) const {
        Options<Args...> opts(argc_, argv_);
        auto t = std::make_tuple(args...);

        try {
            opts.addOptions(t);
            opts.parse(t);
            const auto missing = opts.satisfied();

            bool notSatisfied = !missing.empty();
            if(notSatisfied || opts.needHelp()) {
                std::ostringstream out;
                if (notSatisfied) {
                    out << "Missing options: " << boost::algorithm::join(missing, ", ");
                }

                return makeErrorCode(OptionCategory::MissingRequiredOption,
                                     addDescription(opts, out.str()));
            }

            return error_code_options(opts.parsedOptions());
        } catch (const boost::program_options::unknown_option& err) {
            std::ostringstream out;
            out << "Unknown option \'" << err.get_option_name();
            return makeErrorCode(OptionCategory::UnknownOption,
                                 addDescription(opts, out.str()));
        } catch (const std::exception& err) {
            return makeErrorCode(OptionCategory::UnknownError, err.what());
        }
    }
};

template<class ... Args>
OptionMaker parse(int argc, char** argv) {
    OptionMaker params(argc, argv);
    return params;
}

}
}
}

#endif // MACS_PG_UTILS_SERVICE_OPTIONS_H
