#ifndef MACS_PG_UTILS_SERVICE_PARAMS_H
#define MACS_PG_UTILS_SERVICE_PARAMS_H

#include <vector>
#include <string>
#include <list>
#include <ctime>

namespace macs {
namespace pg {
namespace utils {

enum class Required {
    YES,
    NO
};

enum class IsFlag {
    YES,
    NO
};

template<class T>
struct SingleOption {
    typedef T Type;

    Type& value_;
    Type defaultValue_;

    bool required_;
    bool isFlag_;

    const std::string name_;
    const std::string description_;

    SingleOption(Type& value, Type defVal,
                 const std::string name, const std::string& desc,
                 Required req, IsFlag isFlag)
        : value_(value), defaultValue_(defVal),
          required_(req == Required::YES), isFlag_(isFlag == IsFlag::YES),
          name_(name), description_(desc)
    { }

    SingleOption& def(const Type& t) { defaultValue_ = t; required_ = false; return *this; }
    SingleOption& must() { required_ = true; return *this; }
    SingleOption& notNecessarily() { required_ = false; return *this; }
    void set(const Type& val) { value_ = val; }

    std::string name() const {
        const auto commaIndex = name_.find(',');
        if (commaIndex != std::string::npos) {
            return name_.substr(0, commaIndex);
        }
        return name_;
    }
    std::string fullName() const { return name_; }
    std::string description() const { return description_; }
    const Type& defaultValue() const { return defaultValue_; }
    bool required() const { return required_; }
    bool isFlag() const { return isFlag_; }
};

typedef SingleOption<bool> BoolOption;

#define CREATE_PARAM(ClassName, Name, Descr, Type, DefValue, Required) \
    struct ClassName : SingleOption<Type> { \
    ClassName(Type& val, \
              const std::string& name = Name, \
              const std::string& descr = Descr) \
        : SingleOption<Type>(val, DefValue, name, descr, Required, IsFlag::NO) \
    { } }; \

#define CREATE_FLAG(ClassName, Name, Descr, Required) \
    struct ClassName : BoolOption { \
    ClassName(Type& val, \
              const std::string& name = Name, \
              const std::string& descr = Descr) \
        : SingleOption<Type>(val, false, name, descr, Required, IsFlag::YES) \
    { } }; \

CREATE_PARAM(Uid, "uid", "select mails for user with given id", std::string, "", Required::YES)
CREATE_PARAM(Dsn, "dsn", "database source name", std::string,
         "host=localhost port=5432 user=postgres password=postgres dbname=maildb", Required::NO)

CREATE_PARAM(Fid, "fid", "select mails from folder with given id", std::string, "", Required::YES)
CREATE_PARAM(Mid, "mid", "select mail with given mid", std::string, "", Required::YES)
CREATE_PARAM(Tid, "tid", "select mails from thread with given tid", std::string, "", Required::YES)
CREATE_PARAM(Lid, "lid", "select mails with given label id", std::string, "", Required::YES)
CREATE_PARAM(Stid, "st_id", "select mails with given st_id", std::string, "", Required::YES)
CREATE_PARAM(MsgId, "msg_id", "select mails with given msg_id (message-id)", std::string, "", Required::NO)
CREATE_PARAM(Hid, "hid", "header of the attach", std::string, "", Required::YES)
CREATE_PARAM(ChangeId, "change_id", "change_id param", int64_t, 0, Required::YES)
CREATE_PARAM(SubscriptionId, "subscription_id", "subscription_id param", std::int64_t, 0, Required::YES)
CREATE_PARAM(WorkerId, "worker_id", "worker_id param", std::string, "", Required::YES)
CREATE_PARAM(SubscriptionAction, "subscription_action", "apply action on subscription", std::string, "", Required::NO)
CREATE_PARAM(SubscriptionsLimit, "limit", "subscriptions limit", std::size_t, 0, Required::YES)
CREATE_PARAM(LaunchId, "launch_id", "doberman launch uid", std::string, "", Required::YES)


CREATE_PARAM(VecUint64, "vec_uint64", "simple vector<uint64_t> param", std::vector<uint64_t>, {}, Required::YES)
CREATE_PARAM(Uids, "uids", "select mails for users with given ids", std::vector<std::string>, {}, Required::YES)
CREATE_PARAM(Tids, "tids", "select mails from threads with given tids", std::vector<std::string>, {}, Required::YES)
CREATE_PARAM(Mids, "mids", "select mails with given mids", std::vector<std::string>, {}, Required::YES)
CREATE_PARAM(Fids, "fids", "select mails with given fids", std::vector<std::string>, {}, Required::NO)
CREATE_PARAM(Lids, "lids", "lids for messages", std::vector<std::string>, {}, Required::YES)

CREATE_PARAM(ParentFid, "parent_fid", "create folder in folder with given id", std::string, "", Required::YES)
CREATE_PARAM(FolderName, "folder_name", "create folder with given name", std::string, "", Required::YES)
CREATE_PARAM(FolderPath, "folder_path", "create folder with given path", std::string, "", Required::YES)
CREATE_PARAM(FolderSymbol, "folder_symbol", "create folder with given symbol", std::string, "", Required::YES)

CREATE_PARAM(Int, "int", "simple integer param", int, 0, Required::YES)
CREATE_PARAM(From, "from", "return items start from", int, 0, Required::YES)
CREATE_PARAM(Count, "count", "return given count of items", int, 0, Required::YES)
CREATE_PARAM(SizeT, "size_t", "simple std::size_t param", std::size_t, 0, Required::YES)
CREATE_PARAM(Uint64T, "uint64_t", "simple uint64 param", uint64_t, 0, Required::YES)


CREATE_PARAM(SrcFid, "src_fid", "move messages from folder with given id", std::string, "", Required::YES)
CREATE_PARAM(DstFid, "dst_fid", "move messages to folder with given id", std::string, "", Required::YES)

CREATE_PARAM(WithLid, "with_lid", "select mails in thread with given label", std::string, "", Required::NO)
CREATE_PARAM(WithoutLid, "without_lid", "select mails in thread without given label", std::string, "", Required::NO)

CREATE_PARAM(String, "string", "required string option", std::string, "", Required::YES)

CREATE_PARAM(Time, "timestamp", "unix time", std::time_t, std::time_t(), Required::YES)
CREATE_PARAM(DateFrom, "date_from", "count messages with date newer than it", std::time_t, std::time_t(), Required::NO)
CREATE_PARAM(DateTo, "date_to", "count messages with date older than it", std::time_t, std::time_t(), Required::NO)

CREATE_PARAM(Sort, "sort", "define sort order for messages.\n"
                "Possible values:\n"
                "    received_[descending|ascending]\n"
                "    sent_[descending|ascending]\n"
                "    subject_[descending|ascending]\n"
                "    size_[descending|ascending]\n"
                "    from_[descending|ascending] (now not working)\n"
                "    to_[descending|ascending] (now not working)\n", std::string, "", Required::NO);

CREATE_PARAM(OwnerUid, "owner_uid", "shared folder owner uid", std::string, "", Required::YES)
CREATE_PARAM(OwnerFid, "owner_fid", "shared folder id for owner", std::string, "", Required::YES)
CREATE_PARAM(OwnerRevision, "owner_revision", "revision of owner", uint64_t, 0, Required::YES)
CREATE_PARAM(OwnerMids, "owner_mids", "mids in owner box", std::vector<std::string>, {}, Required::YES)
CREATE_PARAM(OwnerTid, "owner_tid", "Thread id for owner", std::string, "", Required::YES)
CREATE_PARAM(OwnerJoinTids, "owner_join_tids", "Thread ids to join for owner", std::vector<std::string>, {}, Required::YES)

CREATE_PARAM(MaxImapId, "max_imap_id", "max imap_id value", int, 0, Required::YES)
CREATE_PARAM(ChunkSize, "chunk_size", "size of chunks to read", int, 0, Required::YES)

CREATE_PARAM(TokenId, "token_id", "passport token id", std::string, "", Required::YES)
CREATE_PARAM(UidValidity, "uidvalidity", "uidvalidity of imap folder", uint64_t, 0, Required::YES)
CREATE_PARAM(ExternalPath, "external_path", "folder path in external mailbox", std::string, "", Required::YES)
CREATE_PARAM(ShardId, "shard_id", "id of database shard", std::string, "", Required::YES)
CREATE_PARAM(ImapId, "imap_id", "imap id", uint64_t, 0, Required::YES)
CREATE_PARAM(ImapIdFrom, "imap_id_from", "imap id from", uint64_t, 0, Required::YES)
CREATE_PARAM(ImapIdTo, "imap_id_to", "imap id to", uint64_t, 0, Required::YES)
CREATE_PARAM(ImapIds, "imap_ids", "list of imap ids", std::vector<uint64_t>, std::vector<uint64_t>(), Required::YES)
CREATE_PARAM(NewImapIds, "new_imap_ids", "list of new imap ids", std::vector<uint64_t>, std::vector<uint64_t>(), Required::YES)
CREATE_PARAM(MessagesCount, "messages_count", "messages count", uint32_t, 0, Required::YES)
CREATE_PARAM(ErrorsCount, "errors_count", "errors count", uint32_t, 0, Required::YES)
CREATE_PARAM(OauthApp, "oauth_app", "oauth application", std::string, "", Required::NO)
CREATE_PARAM(LastMovedDate, "last_moved_date", "last user moved date", std::time_t, 0, Required::YES)
CREATE_PARAM(LastDeletedDate, "last_deleted_date", "last user deleted date", std::time_t, 0, Required::YES)
CREATE_PARAM(SrcUid, "src_uid", "source uid to collect data from", std::string, "", Required::YES)
CREATE_PARAM(AuthToken, "auth_token", "token for collector authorization", std::string, "", Required::YES)
CREATE_PARAM(RootFolderId, "root_folder_id", "root folder for collector hierarchy", std::string, "0", Required::NO)
CREATE_PARAM(CollectorLabelId, "collector_label_id", "label to put on collected messages", std::string, "0", Required::NO)
CREATE_PARAM(CollectorId, "collector_id", "collector id", int32_t, 0, Required::YES)
CREATE_PARAM(SkippedMids, "skipped_mids", "collector skipped mids ", std::vector<std::string>, {}, Required::YES)
CREATE_PARAM(CollectorState, "collector_state", "collector state", std::string, "", Required::YES)
CREATE_PARAM(CollectorCreationTs, "collector_creation_ts", "creation timestamp of collector", std::time_t, 0, Required::YES)


CREATE_FLAG(Flag, "flag,fl", "simple flag", Required::NO)
CREATE_FLAG(OnlyNew, "only-new,n", "select only new items", Required::NO)
CREATE_FLAG(ThreadsView, "threads-view,t", "select mails grouped by threads", Required::NO)
CREATE_FLAG(WithAttaches, "with-attaches,a", "select only mails with attaches", Required::NO)
CREATE_FLAG(Forced, "forced,f", "forced removal of non-empty folder with subfolders", Required::NO)
CREATE_FLAG(DoUnmark, "unmark,u", "do unmark instead of mark", Required::NO)
CREATE_FLAG(DoUnlabel, "unlabel,u", "do unlabel instead of label", Required::NO)
CREATE_FLAG(StoreDeleted, "store-deleted", "store message in deleted box", Required::NO)

CREATE_PARAM(Tab, "tab", "the type of given tab", std::string, "", Required::YES)

#undef CREATE_FLAG
#undef CREATE_PARAM

}
}
}

#endif // MACS_PG_UTILS_SERVICE_PARAMS_H

