#include "service/get_macs_service.h"
#include "service/options.h"

#include <yamail/data/deserialization/json_reader.h>
#include <internal/envelope/mime_parts.h>

using namespace macs::pg::utils;

int main(int argc, char **argv) {
    std::string uid, dsn, ownerUid, mid, mimeJson;

    const auto parseResult = parse(argc, argv).as(
        Uid(uid), Dsn(dsn), OwnerUid(ownerUid), Mid(mid),
        String(mimeJson, "mime", "mime parts json")
    );

    if (parseResult) {
        std::cerr << parseResult.message() << std::endl;
        return -1;
    }

    using boost::algorithm::join;
    std::cout << "# Sync message for uid=" << uid
              << " with ownerUid=" << ownerUid
              << " mid=" << mid
              << " with mime parts=" << mimeJson << std::endl;

    try {
        std::promise<macs::Envelope> promise;
        auto future = promise.get_future();

        const auto ownerService = getMacsService(ownerUid, dsn);

        ownerService->envelopes().getById(mid, [&](auto err, auto envelope) {
            if (err) {
                mail_errors::setRuntimeError(promise, err);
            } else {
                promise.set_value(envelope);
            }
        });

        auto envelope = future.get();

        auto mimeR = yamail::data::deserialization::fromJson<
            std::vector<macs::pg::reflection::MimePart>>(mimeJson);
        auto mime = macs::pg::toMime(std::move(mimeR));

        const auto service = getMacsService(uid, dsn);
        auto res = service->subscribedFolders().syncMessage(ownerUid, envelope, mime, {}, "");

        std::cout << "mid=" << res.mid << std::endl;
        std::cout << "revision=" << res.revision << std::endl;
        std::cout << "ownerMid=" << res.ownerMid << std::endl;
        std::cout << "ownerRevision=" << res.ownerRevision << std::endl;
    } catch (const std::exception& e) {
        std::cout << "exception: " << e.what() << std::endl;
    }
    return 0;
}
