#include <future>

#include <boost/algorithm/string/join.hpp>
#include <yamail/data/deserialization/json_reader.h>
#include <internal/envelope/mime_parts.h>

#include <service/get_macs_service.h>
#include <service/options.h>

#include <macs/envelope.h>
#include <mail_errors/set_exception.h>
#include "envelope_io.h"

using namespace macs::pg::utils;

int main(int argc, char **argv) {
    using boost::algorithm::join;

    std::string uid, dsn, tid, fid, subject, pop_uidl, st_id,
                firstline, hdr_message_id, extra_data, from, to, mid, mimeJson;
    std::size_t size;
    std::time_t hdr_date, received_date;

    const auto err = parse(argc, argv).as(Mid(mid), Uid(uid), Dsn(dsn),
        Tid(tid), Fid(fid), Stid(st_id),
        Time(hdr_date, "hdr_date", "sent date in unix time"),
        Time(received_date, "received_date", "received date in unix time"),
        String(subject, "subject", "subject string"),
        String(pop_uidl, "pop_uidl", "uidl"),
        String(firstline, "firstline", "firstline string"),
        String(hdr_message_id, "hdr_message_id", "firstline string"),
        SizeT(size, "size", "message size in bytes"),
        String(extra_data, "extra_data", "extra data string"),
        String(mimeJson, "mime", "mime parts json"),
        String(from, "from", "from address"),
        String(to, "to", "to address"));
    if (err) {
        std::cerr << err.message() << std::endl;
        return -1;
    }

    const auto service = getMacsService(uid, dsn);


    std::cout << "#Update envelope for user with uid=" << uid << std::endl;

    try {
        std::promise<void> promise;
        auto result = promise.get_future();
        service->envelopes().getById(mid,
                [&](macs::error_code const & err, const macs::Envelope & old){
            if (err) {
                auto p = std::move(promise);
                mail_errors::setRuntimeError(p, err);
            } else {
                auto mimeR = yamail::data::deserialization::fromJson<
                        std::vector<macs::pg::reflection::MimePart>>(mimeJson);
                auto mime = macs::pg::toMime(std::move(mimeR));
                auto newer = macs::EnvelopeFactory(old).threadId(tid)
                            .fid(fid).date(hdr_date).receiveDate(received_date)
                            .subject(subject).uidl(pop_uidl).stid(st_id)
                            .firstline(firstline).rfcId(hdr_message_id).size(size)
                            .extraData(extra_data).from(from).to(to).mid(mid).release();
                service->envelopes().update(old, std::move(newer), std::move(mime),
                        [] (const macs::Envelope&) { return macs::ThreadMeta {}; },
                        false,
                        [&](macs::error_code err, macs::Envelope e){
                    auto p = std::move(promise);
                    if (err) {
                        mail_errors::setRuntimeError(p, err);
                    } else {
                        EnvelopePrinter(std::cout).print(e);
                        std::cout << "  Revision: " << e.revision() << std::endl;
                        p.set_value();
                    }
                });
            }
        });
        result.get();
    } catch (const std::exception& e) {
        std::cerr << e.what() << std::endl;
    }

    return 0;
}

