# coding: utf-8
from __future__ import print_function
from contextlib import closing
from collections import namedtuple
from itertools import izip_longest
import sys
import os
import subprocess
import threading
import random
from argparse import ArgumentParser
import re
from abc import abstractmethod, ABCMeta

from pyparsing import lineno as find_lineno
from pyparsing import Word, Keyword, alphanums, Optional, \
    OneOrMore, ZeroOrMore, Literal, CharsNotIn, restOfLine, \
    ParseException

import yaml

import logging

log = logging.getLogger()


class Printable(object):
    def __repr__(self):
        fields = ['%s=%r' % o for o in self.__dict__.items()]
        return '%s(%s)' % (
            self.__class__.__name__,
            ', '.join(fields)
        )


class LangParseError(SyntaxError):
    pass


class BaseUtilError(Exception):
    pass


class UnknownVariable(BaseUtilError):
    pass


class MissedAttribute(BaseUtilError):
    pass


class VariableRedefineError(BaseUtilError):
    pass


class UtilExecError(BaseUtilError):
    pass


class VariableDefineError(UtilExecError):
    pass


class VariableEvalError(BaseUtilError):
    pass


class Context(object):
    context = {}

    def __init__(self, rewrite):
        self.rewrite = rewrite

    def get_var(self, name):
        if name not in self.context:
            raise UnknownVariable(
                "Can't find {0} in context".format(
                    name))
        return self.context[name]

    def set_var(self, name, value):
        if not self.rewrite and name in self.context:
            raise VariableRedefineError(
                'Try rewrite {0} old-value: {1} new-value: {2}'.format(
                    name, self.context[name], value))
        self.context[name] = value


class Argument(Printable):
    def __init__(self, name):
        self.name = name


class FlagArgument(Argument):

    def prepare(self):
        return '--' + self.name


class LiteralArgument(Argument):
    def __init__(self, name, val):
        super(LiteralArgument, self).__init__(name)
        self.val = val

    def prepare(self):
        return '--{0.name}={0.val}'.format(self)


class VaribleArgument(Argument, Context):
    def __init__(self, name, var, attr):
        super(VaribleArgument, self).__init__(name)
        self.var = var
        self.attr = attr

    def prepare(self):
        var_value = self.get_var(self.var)
        if self.attr:
            if isinstance(var_value, list):
                var_value = var_value[0]
            try:
                var_value = var_value[self.attr]
            except KeyError as exc:
                raise MissedAttribute(
                    '%s all attributes: %r' % (
                        str(exc), var_value.keys()))
        return '--{0}={1}'.format(self.name, var_value)


class Command(Printable, Context):
    __metaclass__ = ABCMeta

    def __init__(self, new_var, rewrite):
        super(Command, self).__init__(rewrite)
        self.desc = None
        self.lineno = None
        self.new_var = new_var

    def pre_desc(self):
        return self.desc

    @abstractmethod
    def post_desc(self):
        pass

    @abstractmethod
    def error_details(self):
        pass

    @abstractmethod
    def is_util(self):
        pass


class Util(Command):
    base_args = []
    TIMEOUT = 1.5

    def __init__(self, util, arguments, new_var, rewrite):
        super(Util, self).__init__(new_var, rewrite)
        self.util = util
        self.arguments = arguments

        self.out = None
        self.err = None
        self.returncode = None
        self.cmd_args = None
        self.process = None
        self.process_error = None

    def _run_cmd(self):
        try:
            self.process = subprocess.Popen(
                self.cmd_args,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                cwd=os.path.abspath(os.path.dirname(__file__)),
                close_fds=True,
            )
            self.out, self.err = self.process.communicate()
        except OSError as exc:
            self.process_error = UtilExecError(str(exc))

    def run(self):
        self.cmd_args = ['./' + self.util] + self.base_args
        self.cmd_args += [a.prepare() for a in self.arguments]
        log.debug('cmd: %r', self.cmd_args)

        thread = threading.Thread(target=self._run_cmd)
        thread.start()
        thread.join(self.TIMEOUT)
        if thread.is_alive():
            log.warning('terminate process %r command %r', self.process, self)
            self.process.terminate()
            thread.join()
        if self.process_error is not None:
            raise self.process_error
        self.returncode = self.process.returncode
        log.debug(
            'returncode={0.returncode} out={0.out} err={0.err}'.format(
                self))

        if self.returncode:
            raise UtilExecError(
                'Util {0.util} exit with {0.returncode}'.format(self))
        if self.err:
            raise UtilExecError(
                'Util {0.util} write to std::err'.format(self))
        log.debug(self.out)
        if self.new_var:
            try:
                parsed_out = yaml.load(self.out)
            except yaml.YAMLError as exc:
                raise UtilExecError(
                    'Util return malformed out %s' % exc
                )
            log.debug(parsed_out)
            if not parsed_out:
                raise VariableDefineError(
                    'Util {0.util} define emtpy variable {0.new_var} = {1}'.format(
                        self, parsed_out)
                )
            self.set_var(self.new_var, parsed_out)

    def post_desc(self):
        if self.cmd_args:
            desc_args = []
            for a in self.cmd_args:
                if ' ' in a and '=' in a:
                    eq_ind = a.index('=')
                    a = a[:eq_ind+1] + "'" + a[eq_ind+1:] + "'"
                desc_args.append(a)
            return ' '.join(desc_args)
        return ''

    def error_details(self):
        return self.err

    def is_util(self):
        return True


class NewVariable(Command):
    def __init__(self, new_var, rewrite):
        super(NewVariable, self).__init__(new_var, rewrite)
        self.new_var = new_var

    def post_desc(self):
        if self.new_var in self.context:
            return repr(self.get_var(self.new_var))
        return ''

    def error_details(self):
        return ''

    def is_util(self):
        return False

IS_FUNCS = {
    'GREATEST': max,
    'LEAST': min,
}


class FilterCondition(Printable):

    def __init__(self, by_key, is_func, by_value):
        self.by_key = by_key
        self.is_func = is_func
        self.by_value = by_value

    def cmp_by_value(self, value):
        if value is None and (not self.by_value):
            return True
        return str(value) == self.by_value

    def apply(self, from_obj):
        obj = None
        assert from_obj, \
            '%r except non empty obj, got: %r' % (
                self,
                from_obj)
        if self.is_func:
            try:
                max_value = IS_FUNCS[self.is_func](
                    from_obj,
                    key=lambda o: o[self.by_key]
                )[self.by_key]
            except KeyError as exc:
                raise VariableEvalError(
                    '%s available keys: %r' % (
                        str(exc), from_obj[0].keys()))
            obj = [
                o for o in from_obj
                if o[self.by_key] == max_value
            ]
        else:
            compare_values = [self.by_value]
            if self.by_value == '':
                compare_values.append(str(None))
            try:
                obj = [
                    o for o in from_obj if self.cmp_by_value(o[self.by_key])
                ]
            except KeyError as exc:
                raise VariableEvalError(
                    '%s available keys: %r' % (
                        str(exc), from_obj[0].keys()))
        return obj


class Filter(NewVariable):
    def __init__(self, new_var, rewrite, from_var, conditions):
        super(Filter, self).__init__(new_var, rewrite)
        self.from_var = from_var
        self.conditions = conditions

    def run(self):

        obj = self.get_var(self.from_var)
        prev_obj = None
        log.debug(repr(self))
        for cond in self.conditions:
            prev_obj = obj
            obj = cond.apply(obj)
            log.debug('obj=%r prev_obj=%r cond=%r', obj, prev_obj, cond)
            if not obj:
                raise VariableEvalError(
                    '{self.new_var} result is empty: {obj}'
                    ' after {cond} on {prev_obj}'.format(
                        **locals()))
        self.set_var(self.new_var, obj)


class Random(NewVariable):
    def __init__(self, new_var, rewrite, not_in_var, not_in_attr):
        super(Random, self).__init__(new_var, rewrite)
        self.not_in_var = not_in_var
        self.not_in_attr = not_in_attr

    def run(self):
        ignore = set()
        if self.not_in_var:
            obj = self.get_var(self.not_in_var)
            if self.not_in_attr:
                try:
                    obj = [o[self.not_in_attr] for o in obj]
                except KeyError as exc:
                    raise VariableEvalError(
                        '%s available keys: %r' % (
                            str(exc), obj[0].keys()))
            ignore = set(obj)

        with closing(open(__file__)) as fd:
            sample_words = set(re.findall(r'\w{3,}', fd.read()))
            sample_words = list(sample_words - ignore)
            self.set_var(self.new_var, random.choice(sample_words))


def reraise_type_error(func):
    def wrapper(self, toks_str, loc, toks):
        try:
            return func(self, toks_str, loc, toks)
        except TypeError as exc:
            raise RuntimeError(
                'Got TypeError in %r: %s' % (
                    func, str(exc)))
    return wrapper

CommandAndLocation = namedtuple('CommandAndLocation', ['cmd', 'loc'])


class Parser(object):

    def __init__(self):
        self._commands = []
        self._arguments = []
        self._filter_conditions = []

    def _add_cmd(self, obj, loc):
        self._commands.append(
            CommandAndLocation(obj, loc)
        )

    @reraise_type_error
    def _add_comment(self, toks_str, loc, toks):
        self._add_cmd(None, loc)

    @reraise_type_error
    def _add_argument(self, toks_str, loc, toks):
        name = toks.arg_name
        if toks.not_a_flag:
            if toks.var:
                self._arguments.append(
                    VaribleArgument(name, toks.var, toks.var_attr)
                )
            else:
                self._arguments.append(
                    LiteralArgument(name, toks.literal)
                )
        else:
            self._arguments.append(FlagArgument(name))

    @reraise_type_error
    def _add_util(self, toks_str, loc, toks):
        name = toks.util_name
        result = toks.result

        self._add_cmd(
            Util(
                name,
                self._arguments,
                result,
                rewrite=toks.define_style == 'REDEFINE'),
            loc
        )
        self._arguments = []

    @reraise_type_error
    def _add_new_filter(self, toks_str, loc, toks):
        self._add_cmd(
            Filter(
                new_var=toks.new_var,
                rewrite=toks.var_def_style == 'REDEFINE',
                from_var=toks.from_var,
                conditions=self._filter_conditions,
                ),
            loc)
        self._filter_conditions = []

    @reraise_type_error
    def _add_new_condition(self, toks_str, loc, toks):
        self._filter_conditions.append(
            FilterCondition(
                toks.by_key,
                toks.is_func,
                toks.by_value))

    @reraise_type_error
    def _add_random(self, toks_str, loc, toks):
        self._add_cmd(
            Random(
                new_var=toks.new_var,
                rewrite=toks.var_def_style == 'REDEFINE',
                not_in_var=toks.not_in_var,
                not_in_attr=toks.not_in_attr),
            loc
        )

    def _make_gramma(self):
        lower_alphanums = ''.join([a for a in alphanums if a.lower() == a])
        statement_end = Literal(';')
        arg_name = Word(lower_alphanums+'-_').setResultsName('arg_name')
        variable = Word('$', alphanums+'-').setResultsName('var')
        variable_attr = Word(alphanums+'_-')
        mk_literal = lambda name: (
            Word(alphanums+'-_.*+').setResultsName(name) |
            (Literal('"') +
             Optional(CharsNotIn('"')).setResultsName(name) +
             Literal('"'))
        )

        flag_arg = arg_name
        arg_literal = mk_literal('literal')
        arg_variable = \
            variable + \
            Optional(
                Literal('.') + variable_attr.setResultsName('var_attr'))
        arg = arg_name + \
            Optional(
                Literal('=') + (arg_literal | arg_variable)
            ).setResultsName('not_a_flag')

        util_argument = (arg | flag_arg)
        util_argument.addParseAction(self._add_argument)

        util_define_style = (
            Keyword('DEFINE') | Keyword('REDEFINE')
        ).setResultsName('define_style')

        util = (
            Word(alphanums, alphanums + '_').setResultsName('util_name') +
            Optional(Literal('(') + OneOrMore(util_argument) + Literal(')')) +
            Optional(util_define_style + (variable.setResultsName('result'))) +
            statement_end
        )
        util.addParseAction(self._add_util)

        var_def_style = (
            Keyword('DEFINE')|Keyword('REDEFINE')
        ).setResultsName('var_def_style')
        base_var_def = (
            var_def_style +
            variable.setResultsName('new_var') +
            Keyword('FROM'))

        filter_by_value = (
            Keyword('=') + mk_literal('by_value')
        )

        filter_by_is_func = (
            Keyword('IS') +
            (Keyword('GREATEST') | Keyword('LEAST')).setResultsName('is_func'))

        filted_condition = (
            variable_attr.setResultsName('by_key') +
            (filter_by_value | filter_by_is_func))
        filted_condition.addParseAction(self._add_new_condition)

        filter_gr = (
            base_var_def +
            variable.setResultsName('from_var') +
            Keyword('WHERE') +
            filted_condition +
            ZeroOrMore(Keyword('AND') + filted_condition) +
            statement_end)
        filter_gr.addParseAction(self._add_new_filter)

        radom_not_in = (
            Keyword('NOT') +
            Keyword('IN') +
            variable.setResultsName('not_in_var') +
            Optional(
                Literal('.') + variable_attr.setResultsName('not_in_attr'))
            )
        random_gr = (
            base_var_def +
            Keyword('RANDOM') +
            Optional(radom_not_in) +
            statement_end
        )
        random_gr.addParseAction(self._add_random)

        comment_line = Literal('#') + restOfLine
        comment_line.addParseAction(self._add_comment)

        return OneOrMore(filter_gr | random_gr | util | comment_line)

    def _set_desc(self, lang_text):
        commands = []
        for cl, next_cl in izip_longest(self._commands, self._commands[1:]):
            # ignore comments
            if cl.cmd is None:
                continue
            desc = ''
            # last command
            if next_cl is None:
                desc = lang_text[cl.loc:]
            else:
                desc = lang_text[cl.loc:next_cl.loc]
            cmd = cl.cmd
            cmd.desc = desc.strip()
            cmd.lineno = find_lineno(cl.loc, lang_text)
            commands.append(cmd)
        return commands

    def parse(self, lang_text):
        g = self._make_gramma()
        try:
            g.parseString(lang_text, parseAll=True)
        except ParseException as exc:
            for i, line in enumerate(lang_text.split('\n'), 1):
                if exc.lineno - 3 < i < exc.lineno + 3:
                    if i == (exc.lineno):
                        print('{0:<3}*'.format(i), hl_error(line))
                    else:
                        print('{0:<4}'.format(i), line)
            raise LangParseError(exc)
        return self._set_desc(lang_text)


def termcode(num):
    return '\033[%sm' % num

hl_error = hl_error = lambda err: err

if sys.stdout.isatty() or os.environ.get('FORCE_COLORS'):
    hl_error = lambda err: termcode(31) + err + termcode(0)
    hl_ok = lambda err: termcode(32) + err + termcode(0)


def parse_file(file_name):
    with closing(open(file_name)) as fd:
        lang_text = fd.read()
        p = Parser()
        return p.parse(lang_text)


def print_shift(text):
    for line in text.split('\n'):
        print(' '*4 + line)


def find_all_utils():
    all_utils = []
    with closing(open(
        os.path.join(
            os.path.dirname(__file__),
            'CMakeLists.txt'))) as fd:
        start_block = False
        for line in fd:
            if 'COMMAND bash utils' in line:
                break
            if start_block:
                line_utils = line.strip().split(' ')
                all_utils += [u for u in line_utils if u]
            elif line.startswith('add_custom_target(utils'):
                start_block = True
    return all_utils


def main():
    parser = ArgumentParser(
        description='initialize data for utils check',
    )
    parser.add_argument(
        '-f', '--file',
        default=os.path.join(
            os.path.dirname(__file__),
            'utils_lang.txt')
    )
    parser.add_argument(
        '--dsn',
        metavar='PG_DSN',
        type=str,
        default='postgresql://ymail:ympass@/mpi'
    )
    parser.add_argument(
        '--uid',
        metavar='UID',
        default="42"
    )
    parser.add_argument(
        '-v', '--verbose',
        action='count',
        help='be verbose',
        default=0,
    )
    parser.add_argument(
        '-s', '--stop',
        action='store_true',
        help='exit on first failure'
    )
    parser.add_argument(
        '--find-untested',
        action='store_true',
        help='find untested utils'
    )

    args = parser.parse_args()

    logging.basicConfig(
        level=logging.WARNING - 10*(min(2, args.verbose)),
        file=sys.stderr,
        format='%(asctime)-15s %(filename)s '
               '+%(lineno)d %(levelname)s: %(message)s'
    )
    Util.base_args = ['--uid=%s' % args.uid, '--dsn=%s' % args.dsn]
    utils_count = 0
    fail_count = 0
    try:
        commands = parse_file(args.file)
    except LangParseError as exc:
        print(str(exc))
        return -1
    tested_utils = set()
    for cmd in commands:
        try:
            if cmd.is_util():
                utils_count += 1
                tested_utils.add(cmd.util)
            cmd.run()
            print('{0:<4}{1}  {2}'.format(
                cmd.lineno, hl_ok(cmd.pre_desc()), cmd.post_desc()))
        except BaseUtilError as exc:
            if cmd.is_util():
                fail_count += 1
            print('{0:<4}{1}  {2}'.format(
                cmd.lineno, hl_error(cmd.pre_desc()), cmd.post_desc()))
            print_shift('{0.__class__.__name__}:{0}'.format(exc))
            if cmd.error_details():
                print_shift(cmd.error_details())
            if args.stop:
                break
    print('Total utils : %d\nFailed utils: %d' % (utils_count, fail_count))
    if args.find_untested:
        all_utils = set(find_all_utils())
        untested_utils = all_utils - tested_utils
        print('Untested utils: %d' % len(untested_utils))
        print('\n'.join(untested_utils))
    return fail_count


if __name__ == '__main__':
    main()
