#pragma once

#include <boost/system/error_code.hpp>
#include <boost/system/system_error.hpp>

namespace mail_errors {

template <typename ErrorCodeEnum, typename T>
using enable_if_error_enum = std::enable_if_t<
        boost::system::is_error_code_enum<ErrorCodeEnum>::value, T>;

class error_code {
public:
    using base_type = ::boost::system::error_code;
    using error_category = ::boost::system::error_category;
    using error_condition = decltype(base_type().default_error_condition());

    error_code() {}

    error_code(int val, const error_category & cat ) : base_(val, cat) {}
    error_code(int val, const error_category & cat, std::string what)
    : base_(val, cat), what_(std::move(what)) {}

    error_code(base_type e) : base_(std::move(e)) {}
    error_code(base_type e, std::string what)
    : base_(std::move(e)), what_(std::move(what)) {}

    template <class ErrorCodeEnum>
    explicit error_code(ErrorCodeEnum e,
            enable_if_error_enum<ErrorCodeEnum, void>* = nullptr)
        : base_(make_error_code(e)) {}

    template <class ErrorCodeEnum>
    explicit error_code(ErrorCodeEnum e, std::string what,
            enable_if_error_enum<ErrorCodeEnum, void>* = nullptr)
        : base_(make_error_code(e)), what_(std::move(what)) {}

    error_condition default_error_condition() const noexcept {
        return base().default_error_condition();
    }
    std::string message() const {
        return what().empty() ? base().message() : what();
    }
    const std::string & what() const noexcept { return what_; }

    std::string full_message() const {
        using namespace std::string_literals;
        std::string result = base().message();
        if (!what().empty()) {
            result += ": "s + what();
        }
        return result;
    }

    int value() const noexcept { return base().value(); }
    const error_category & category() const noexcept { return base().category(); }

    explicit operator bool() const noexcept { return static_cast<bool>(base());}
    bool operator!() const noexcept { return !static_cast<bool>(base());}

    const base_type & base() const noexcept { return base_;}

private:
    base_type base_;
    std::string what_;
};

using error_condition = error_code::error_condition;

inline bool operator==( const error_code & lhs, const error_code & rhs ) noexcept {
    return lhs.base() == rhs.base();
}

inline bool operator==( const error_code & lhs, const error_condition & rhs ) noexcept {
    return lhs.base() == rhs;
}

inline bool operator==( const error_code & lhs, const boost::system::error_code & rhs ) noexcept {
    return lhs.base() == rhs;
}

inline bool operator==( const error_condition & lhs, const error_code & rhs ) noexcept {
    return lhs == rhs.base();
}

inline bool operator==( const boost::system::error_code & lhs, const error_code & rhs ) noexcept {
    return lhs == rhs.base();
}

inline bool operator!=( const error_code & lhs, const error_code & rhs ) noexcept {
    return !(lhs == rhs);
}

inline bool operator!=( const error_code & lhs, const error_condition & rhs ) noexcept {
    return !(lhs == rhs);
}

inline bool operator!=( const error_code & lhs, const boost::system::error_code & rhs ) noexcept {
    return !(lhs == rhs);
}

inline bool operator!=( const error_condition & lhs, const error_code & rhs ) noexcept {
    return !(lhs == rhs);
}

inline bool operator!=( const boost::system::error_code & lhs, const error_code & rhs ) noexcept {
    return !(lhs == rhs);
}

inline bool operator<( const error_code & lhs, const error_code & rhs ) noexcept {
    return lhs.base() < rhs.base();
}

inline std::size_t hash_value( const error_code & ec ) {
  return hash_value(ec.base());
}
class system_error : public ::boost::system::system_error {
    using base = ::boost::system::system_error;
public:
    system_error(error_code ec) : base(ec.base(), ec.what()) {}
    system_error(error_code ec, std::string msg) : base(ec.base(), msg + ec.what()) {}
};

} // namespace mail_errors
