#pragma once

#include <memory>
#include <internal/storage/handlers.h>
#include <mail_getter/io.h>

namespace mail_getter {
namespace storage {

class Service {
public:
    virtual ~Service() = default;

    template <typename Handler = io::sync_context>
    auto getBlob(const Stid& stid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnGetBlob> init(handler);
        asyncGetBlob(stid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto putBlob(const std::string& baseId, const std::string& data, std::chrono::seconds ttl,
            Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnPutBlob> init(handler);
        asyncPutBlob(baseId, data, ttl, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getXml(const Stid& stid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnGetXml> init(handler);
        asyncGetXml(stid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getByRange(const Stid& stid, const Range& range, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnGetBlob> init(handler);
        asyncGetByRange(stid, range, init.handler);
        return init.result.get();
    }
protected:
    virtual void asyncGetBlob(const Stid& stid, OnGetBlob handler) const = 0;
    virtual void asyncPutBlob(const std::string& baseId, const std::string& data, std::chrono::seconds ttl,
            OnPutBlob handler) const = 0;
    virtual void asyncGetXml(const Stid& stid, OnGetXml handler) const = 0;
    virtual void asyncGetByRange(const Stid& stid, const Range& range, OnGetBlob handler) const = 0;
};

using ServicePtr = std::shared_ptr<Service>;

}  // namespace storage
}  // namespace mail_getter
