#pragma once

#include <map>
#include <chrono>
#include <butil/crypt.h>
#include <mail_getter/part_id.h>

namespace mail_getter::attach_sid {

struct KeyContainer {
    std::map<std::string, crypto::AesKey> aesKeys;
    std::map<std::string, crypto::HmacKey> hmacKeys;
};

struct Keys {
    Keys() = default;

    Keys(const KeyContainer& container, std::string aesKeyId_, std::string hmacKeyId_)
        : aesKeyId(std::move(aesKeyId_))
        , hmacKeyId(std::move(hmacKeyId_))
    {
        const auto verifyKey = [] (const auto& keys, const std::string& keyId, const std::string& keyName) {
            auto key = keys.find(keyId);
            if (keys.end() == key) {
                throw std::invalid_argument("couldn't find " + keyName + " key for attach_sid with id " + keyId);
            }

            return key->second;
        };
        aesKey = verifyKey(container.aesKeys, aesKeyId, "aes");
        hmacKey = verifyKey(container.hmacKeys, hmacKeyId, "hmac");
    }

    std::string aesKeyId;
    crypto::AesKey aesKey;
    std::string hmacKeyId;
    crypto::HmacKey hmacKey;
};

std::time_t timestamp();

struct Packer {
    Packer(std::chrono::seconds timeout, const Keys& keys)
        : _expirationTime(timestamp() + timeout.count())
        , _iv(crypto::generateIv(16))
        , _keys(keys)
    {}

    Packer(std::time_t baseTime, std::chrono::seconds timeout, const Keys& keys, crypto::blob iv)
        : _expirationTime(baseTime + timeout.count())
        , _iv(std::move(iv))
        , _keys(keys)
    {}

    std::string operator()(const part_id::Old& partId) const;
    std::string operator()(part_id::Temporary partId) const;
    std::string operator()(part_id::SingleMessagePart partId) const;
    std::string operator()(part_id::MultipleMessagePart partId) const;

private:
    std::time_t _expirationTime;
    const crypto::blob _iv;
    const Keys& _keys;
};

struct Unpacker {
    explicit Unpacker(const KeyContainer& keyContainer)
        : _keyContainer(keyContainer)
    {}

    part_id::Variant operator()(const std::string& sid) const;

private:
    const KeyContainer& _keyContainer;
};


struct InvalidSid : public std::runtime_error {
    using std::runtime_error::runtime_error;
};

struct OldSidFormat : public InvalidSid {
    using InvalidSid::InvalidSid;
};

struct IllformedSid : public InvalidSid {
    using InvalidSid::InvalidSid;
}; 

struct ExpiredSid : public InvalidSid {
    using InvalidSid::InvalidSid;
};

struct InvalidAesKeyId : public InvalidSid {
    using InvalidSid::InvalidSid;
};

struct InvalidHmacKeyId : public InvalidSid {
    using InvalidSid::InvalidSid;
};

struct DecryptionError : public InvalidSid {
    using InvalidSid::InvalidSid;
};

void parseSid(const std::string &decryptedSid, std::time_t& ts, std::string &mid, std::string &hid);

} // namespace mail_getter::attach_sid
