#pragma once

#include <memory>
#include <unordered_map>
#include <mail_getter/types.h>
#include <mail_getter/MessageAccessDefinitions.h>
#include <macs/envelope.h>
#include <macs/hooks.h>

namespace macs {

class EnvelopesRepository;

} // namespace macs

namespace mail_getter {

struct MessageAccessParams {
    Stid stid;
    MetaParts parts;
};

struct UnknownMid : std::runtime_error {
    using std::runtime_error::runtime_error;
};

using MessageAccessParamsByHid = std::unordered_map<Hid, MessageAccessParams>;

struct MessageAccessWithWindatParams {
    MessageAccessParams mainParams;
    MessageAccessParamsByHid windatParams;
};

using MidStidMime = macs::MidsWithMimes::value_type;

MessageAccessParams getMessageAccessParams(macs::Stid stid);
MessageAccessParams getMessageAccessParams(macs::Stid stid, macs::MimeParts mimeParts);
MessageAccessParams getMessageAccessParams(MidStidMime data);

template<class Envelopes>
MessageAccessParams getMessageAccessParams(
        const Envelopes& envelopes, macs::Mid mid, bool withDeleted = false) {
    if (mid.empty()) {
        throw std::invalid_argument("getMessageAccessParams error: empty mid");
    }

    const macs::Mids mids{std::move(mid)};
    auto mimes = withDeleted ? envelopes.getMimesWithDeleted(mids) : envelopes.getMimes(mids);
    if (mimes.size() == 0) {
        throw UnknownMid("getMessageAccessParams error: unknown mid=" + mids.front());
    } else if (mimes.size() > 1) {
        throw std::logic_error("getMimes returns more then one record for one mid=" + mids.front());
    }

    return getMessageAccessParams(std::move(mimes[0]));
}

template<class Envelopes, class Yield>
MessageAccessParams getMessageAccessParams(
        const Envelopes& envelopes, macs::Mid mid, Yield ctx) {
    if (mid.empty()) {
        throw std::invalid_argument("getMessageAccessParams error: empty mid");
    }

    const macs::Mids mids{std::move(mid)};
    auto mimes = envelopes.getMimes(mids, ctx);
    if (mimes.size() == 0) {
        throw UnknownMid("getMessageAccessParams error: unknown mid=" + mids.front());
    } else if (mimes.size() > 1) {
        throw std::logic_error("getMimes returns more then one record for one mid=" + mids.front());
    }

    return getMessageAccessParams(std::move(mimes[0]));
}

template<class Envelopes>
std::tuple<MessageAccessParams, macs::Attachments> getMessageAccessParamsWithAttaches(
        const Envelopes& envelopes, macs::Mid mid, bool withDeleted = false) {
    if (mid.empty()) {
        throw std::invalid_argument("getMessageAccessParams error: empty mid");
    }

    mail_errors::error_code ec;
    const macs::Mids mids{std::move(mid)};
    macs::MidsWithMimesAndAttaches mimes = withDeleted ?
            envelopes.getMimesWithAttachesWithDeleted(mids, ec) : envelopes.getMimesWithAttaches(mids, ec);
    if (ec) {
        throw std::runtime_error(ec.what());
    } else if (mimes.size() == 0) {
        throw UnknownMid("getMessageAccessParams error: unknown mid=" + mids.front());
    } else if (mimes.size() > 1) {
        throw std::logic_error("getMimes returns more then one record for one mid=" + mids.front());
    }

    auto mainParams = std::make_tuple(std::get<0>(mimes[0]), std::get<1>(mimes[0]), std::get<2>(mimes[0]));
    return std::make_tuple(getMessageAccessParams(std::move(mainParams)), std::get<3>(mimes[0]));
}

template<class Envelopes, class Yield>
MessageAccessWithWindatParams getMessageAccessWithWindatParams(
        const Envelopes& envelopes, macs::Mid mid, Yield ctx = macs::io::use_sync) {
    auto mainParams = getMessageAccessParams(envelopes, mid, ctx);

    MessageAccessParamsByHid windatParams;
    auto windatMimes = envelopes.getWindatMimes({mid}, ctx);
    for (auto &midStidWithMime : windatMimes) {
        auto &mimes = std::get<2>(midStidWithMime);

        if (mimes.empty()) {
            throw UnknownMid("getMessageAccessParams error: bad windat mime for mid=" + mid);
        }
        auto hid = mimes[0].hid();
        windatParams.insert( {std::move(hid), getMessageAccessParams(std::move(midStidWithMime))} );
    }

    return { std::move(mainParams), std::move(windatParams) };
}

}  // namespace mail_getter
