#ifndef ALIAS_CLASS_LIST_H
#define ALIAS_CLASS_LIST_H

#include <string>
#include <map>
#include <set>

#include <boost/property_tree/ptree_fwd.hpp>

#include <mail_getter/mime_type.h>

/**
 * Provides mappings between mime types and
 * short human-readable strings.
 * Fixes DARIA-30
 */
class AliasClassList
{
public:
    AliasClassList() {}
    explicit AliasClassList(const std::string& config);
    /**
     * Const reference will be valid only after calls to const methods.
     * @return alias of mime-type.
     */
    const std::string& getMimeAlias(const MimeType& mType) const;
    /**
     * @return alias of extension.
     */
    const std::string& getExtAlias(const std::string& ext) const;
    const std::string& getAlias(const std::string& ext, const MimeType& mType) const;

    bool canBeThumbnailed(const MimeType& mType) const;
    bool canBeRotated(const MimeType& mType) const;
    bool viewForbidden(const MimeType& mType) const;
    bool canBeBrowsed(const MimeType& mType) const;

    static const std::string& defaultAlias();

private:
    void parseTypeNode(const boost::property_tree::ptree& typeNode);
    void parseMimeNode(const boost::property_tree::ptree& mimeNode, const std::string& alias);
    void parseMimeAttr(const boost::property_tree::ptree& mimeAttr, const std::string& mimeType);
    void parseExtNode(const boost::property_tree::ptree& extNode, const std::string& alias);

private:
    typedef std::map<std::string, std::string> AliasMap;
    /**
     * Map consisist of:
     * "type/subtype" => "tag" [Means that subtype has a tag]
     * "type/" => "tag" [Means that type/[any subtype] have that tag]
     */
    AliasMap mimeTypesAliasMap_;
    /**
     * ext => alias
     */
    AliasMap extAliasMap_;

    typedef std::set<std::string> FlagSet;
        /**
         * Set consisist of:
         * "type/subtype" => here / not here [Means that subtype can / can't be thumbnailed]
         * "type/" => here / not here [Means that type/[any subtype] can / can't be thumbnailed]
         */
    FlagSet mimeTypesThumbSet_;
    FlagSet mimeTypesRotatableSet_;
    FlagSet mimeTypesNoViewSet_;
        /**
         * Set consisist of:
         * "type/subtype" => here / not here [Means that subtype can / can't be shown in browser]
         * "type/" => here / not here [Means that type/[any subtype] can / can't be shown in browser]
         */
    FlagSet mimeTypesBrowserSet_;
};

#endif
